<?php
namespace WCStepFilter;

/**
 * Filter Class
 *
 * @class Filter
 * @version 5.6.1
 */
class Filter
{
    // <editor-fold desc="Properties">
    /**
     * Settings model variable
     *
     * @var array
     */
    public static $settingsModel;

    /**
     * Active steps session keys variable
     *
     * @var string
     */
    public static $activeStepsSessionKey = 'wcsf-active-step';

    /**
     * Filter value session keys variable
     *
     * @var string
     */
    public static $valuesSessionKey = 'wcsf-values';

    /**
     * Notices array
     * @var array
     */
    public $notices = [];
    // </editor-fold>

    // <editor-fold desc="Core">
    /** Constructor */
    public function __construct()
    {
        add_action('init', [$this, 'onInit']);

        // admin scripts enqueue
        add_action('admin_enqueue_scripts', [$this, 'enqueueAdmin']);

        // post fields
        add_action('add_meta_boxes', [$this, 'addMetaBoxes']);
        add_action('save_post_wcsf', [$this, 'savePostAction']);

        // admin columns
        add_filter('manage_wcsf_posts_columns', [$this, 'postColumnsFilter']);
        add_action('manage_wcsf_posts_custom_column', [$this, 'postCustomColumnFilter'], 10, 2);

        // ajax
        add_action('wp_ajax_nopriv_WCSFGetStep', [$this, 'getStepHtmlAjax']);
        add_action('wp_ajax_WCSFGetStep', [$this, 'getStepHtmlAjax']);
        add_action('wp_ajax_nopriv_WCSFSubmitAndGoTo', [$this, 'submitAndGoToAjax']);
        add_action('wp_ajax_WCSFSubmitAndGoTo', [$this, 'submitAndGoToAjax']);
        add_action('wp_ajax_nopriv_WCSFSubmitAndGoNext', [$this, 'submitAndGoNextAjax']);
        add_action('wp_ajax_WCSFSubmitAndGoNext', [$this, 'submitAndGoNextAjax']);
        add_action('wp_ajax_nopriv_WCSFReset', [$this, 'resetAjax']);
        add_action('wp_ajax_WCSFReset', [$this, 'resetAjax']);
    }

    /** On init action */
    public function onInit()
    {
        $defaultResultsURL = '/';

        if (function_exists('wc_get_page_id')) {
            $defaultResultsURL = str_replace(home_url(), '', get_permalink(wc_get_page_id('shop')));
        }

        self::$settingsModel = apply_filters(
            'wcsf_settings_model',
            [
                // <editor-fold desc="Basic">
                'questions' => [
                    'label' => L10N::r('Select questions for using'),
                    'key' => '_questions',
                    'type' => 'multi-select',
                    'default' => [],
                    'values' => Question::getPostsIdsList(),
                    'group' => L10N::r('Basic')
                ],
                // </editor-fold>
                // <editor-fold desc="Results">
                'results_url' => [
                    'label' => L10N::r('Results URL'),
                    'key' => '_results_url',
                    'type' => 'text',
                    'default' => $defaultResultsURL,
                    // phpcs:disable
                    'description' => L10N::r('Set # to stay on the same page (for example, if you are using the filter as a widget)'),
                    // phpcs:enable
                    'group' => L10N::r('Results')
                ],
                'show_results_within' => [
                    'label' => L10N::r('Show results within the filter'),
                    'key' => '_show_results_within',
                    'type' => 'checkbox',
                    'default' => false,
                    'group' => L10N::r('Results')
                ],
                'results_tab_title' => [
                    'label' => L10N::r('Results tab title'),
                    'key' => '_results_tab_title',
                    'default' => L10N::r('Results'),
                    'type' => 'text',
                    'group' => L10N::r('Results')
                ],
                // </editor-fold>
                // <editor-fold desc="Behavior">
                'scrolling_top_on_update' => [
                    'label' => L10N::r('Scrolling to top on the form update'),
                    'key' => '_scrolling_top_on_update',
                    'type' => 'checkbox',
                    'default' => true,
                    'public' => true,
                    'group' => L10N::r('Behavior')
                ],
                'scrolling_up_gap' => [
                    'label' => L10N::r('The gap on scrolling up (px)'),
                    'key' => '_scrolling_up_gap',
                    'type' => 'number',
                    'default' => 0,
                    'public' => true,
                    'group' => L10N::r('Behavior')
                ],
                // </editor-fold>
                // <editor-fold desc="Controls">
                'back_button_text' => [
                    'label' => L10N::r('"Back" button text'),
                    'key' => '_back_button_text',
                    'default' => L10N::r('Back'),
                    'type' => 'text',
                    'group' => L10N::r('Controls')
                ],
                'skip_button_text' => [
                    'label' => L10N::r('"Skip" button text'),
                    'key' => '_skip_button_text',
                    'default' => L10N::r('Skip'),
                    'type' => 'text',
                    'group' => L10N::r('Controls')
                ],
                'next_button_text' => [
                    'label' => L10N::r('"Next" button text'),
                    'key' => '_next_button_text',
                    'default' => L10N::r('Next'),
                    'type' => 'text',
                    'group' => L10N::r('Controls')
                ],
                'reset_button_text' => [
                    'label' => L10N::r('"Reset" button text'),
                    'key' => '_reset_button_text',
                    'default' => L10N::r('Reset'),
                    'type' => 'text',
                    'group' => L10N::r('Controls')
                ],
                'results_button_text' => [
                    'label' => L10N::r('"Results" button text'),
                    'key' => '_results_button_text',
                    'default' => L10N::r('Show results'),
                    'type' => 'text',
                    'group' => L10N::r('Controls')
                ]
                // </editor-fold>
            ]
        );

        $name = 'Step Filter';
        $args = [
            'label' => $name,
            'labels' => [
                'name' => $name,
                'singular_name' => $name,
                'menu_name' => $name
            ],
            'description' => L10N::r('This is where you can add new step filter items'),
            'public' => false,
            'show_ui' => true,
            'map_meta_cap' => true,
            'publicly_queryable' => false,
            'exclude_from_search' => true,
            'show_in_menu' => current_user_can('manage_woocommerce') ? 'woocommerce' : true,
            'hierarchical' => false,
            'rewrite' => false,
            'query_var' => false,
            'supports' => [
                'title',
                'editor'
            ],
            'show_in_nav_menus' => false,
            'show_in_admin_bar' => true
        ];

        register_post_type('wcsf', $args);
    }
    // </editor-fold>

    // <editor-fold desc="Admin">
    /** Admin styles and scripts enqueue */
    public function enqueueAdmin()
    {
        if (defined('WC_VERSION')
            && ((isset($_GET['post']) && get_post_type((int) $_GET['post']) == 'wcsf')
                || (isset($_GET['post_type']) && $_GET['post_type'] == 'wcsf'))
        ) {
            wp_register_script(
                'select2',
                \WC()->plugin_url() . '/assets/js/select2/select2.full.min.js',
                ['jquery'],
                '4.0.3'
            );

            wp_register_script(
                'selectWoo',
                WC()->plugin_url() . '/assets/js/selectWoo/selectWoo.full.min.js',
                ['jquery'],
                '1.0.0'
            );

            wp_register_script(
                'wc-enhanced-select',
                WC()->plugin_url() . '/assets/js/admin/wc-enhanced-select.min.js',
                ['jquery', 'selectWoo'],
                WC_VERSION
            );

            wp_enqueue_script('select2');
            wp_enqueue_script('selectWoo');
            wp_enqueue_script('wc-enhanced-select');
            wp_enqueue_style('woocommerce_admin_styles', \WC()->plugin_url() . '/assets/css/admin.css', [], WC_VERSION);
        }
    }

    /**
     * Save meta values
     *
     * @param integer $postId
     */
    public function savePostAction($postId)
    {
        if ((defined('DOING_AUTOSAVE') && DOING_AUTOSAVE)
            || !current_user_can('edit_page', $postId)
            || (isset($_REQUEST['action']) && $_REQUEST['action'] == 'inline-save')
        ) {
            return;
        }

        foreach (self::$settingsModel as $setting) {
            $value = '';

            if (isset($_POST[$setting['key']])) {
                $value = $_POST[$setting['key']];
            }

            update_post_meta($postId, $setting['key'], $value);
        }
    }

    /** Add meta-boxes */
    public function addMetaBoxes()
    {
        add_meta_box(
            'woocommerce-step-filter',
            L10N::r('Settings'),
            [$this, 'metaBoxCallback'],
            'wcsf',
            'normal'
        );
    }

    /**
     * Post page meta-box content
     *
     * @param object $post
     */
    public function metaBoxCallback($post)
    {
        $postSettingsModelGroups = [];
        $shortCodes = [
            'shortcode' => [
                'group' => L10N::r('Basic'),
                'label' => L10N::r('Shortcode for using'),
                'key' => '_shortcode',
                'type' => 'text',
                'readonly' => true,
                'default' => '[woocommerce-step-filter id="' . $post->ID . '"]'
            ]
        ];

        self::$settingsModel = $shortCodes + self::$settingsModel;

        foreach (self::$settingsModel as $settingKey => $setting) {
            $postSettingsModelGroups[$setting['group']][$settingKey] = $setting;
        }
        ?>
        <section class="wcsf-filter wcsf-settings-group" data-component="wcsf-filter wcsf-settings-group">
            <?php foreach ($postSettingsModelGroups as $group => $settingsModel) { ?>
                <fieldset>
                    <a class="button button-large wcsf-settings-group-toggle" href="#" role="button"
                        data-component="wcsf-settings-group-toggle"
                        data-id="<?php echo esc_attr($group); ?>"><?php echo wp_kses_post($group); ?></a>
                    <table class="form-table wcsf-settings-table wcsf-settings-group-content<?php
                        echo reset($postSettingsModelGroups) == $settingsModel ? ' is-visible' : '';
                        ?>"
                        data-component="wcsf-settings-group-content"
                        data-id="<?php echo esc_attr($group); ?>">
                        <?php foreach ($settingsModel as $modelItemKey => $modelItem) { ?>
                            <tr data-component="wcsf-setting" data-key="<?php echo esc_attr($modelItemKey); ?>">
                                <th>
                                    <label for="<?php echo esc_attr($modelItem['key']); ?>"><?php
                                        echo wp_kses_post($modelItem['label']);
                                        ?></label>
                                </th>
                                <td><?php
                                    Core::settingFieldView(
                                        $modelItem,
                                        ['values' => [$modelItem['key'] => self::getSetting($post->ID, $modelItemKey)]]
                                    );
                                    ?></td>
                            </tr>
                        <?php } ?>
                    </table>
                </fieldset>
            <?php } ?>
        </section>
        <?php
    }

    /**
     * Filters list columns filter
     *
     * @param array $columns
     *
     * @return array $columns
     */
    public function postColumnsFilter($columns)
    {
        $columns['shortcode'] = L10N::r('ShortCode');

        return $columns;
    }

    /**
     * Filters list line cell
     *
     * @param array $columns
     * @param integer $postId
     */
    public function postCustomColumnFilter($columns, $postId)
    {
        if ($columns == 'shortcode') {
            echo '[woocommerce-step-filter id="' . (int) $postId . '"]';
        }
    }
    // </editor-fold>

    // <editor-fold desc="Settings">
    /**
     * Get one of post settings
     *
     * @param integer $id
     * @param string $setting
     *
     * @return string|boolean|array|float
     */
    public static function getSetting($id, $setting)
    {
        static $filterSettingsCache;

        if (isset($filterSettingsCache[$id], $filterSettingsCache[$id][$setting])) {
            return apply_filters('wcsf_setting', $filterSettingsCache[$id][$setting], $id, $setting);
        }

        $value = isset(self::$settingsModel[$setting]['key'])
            ? get_post_meta($id, self::$settingsModel[$setting]['key'], true)
            : '';

        if ($value == '' && isset(self::$settingsModel[$setting]['default'])) {
            $value = self::$settingsModel[$setting]['default'];
        }

        $value = Core::handleSettingType($value, self::$settingsModel[$setting]['type']);

        if (!isset($filterSettingsCache[$id])) {
            $filterSettingsCache[$id] = [];
        }

        $filterSettingsCache[$id][$setting] = $value;

        return apply_filters('wcsf_setting', $value, $id, $setting);
    }

    /**
     * Get post settings
     *
     * @param integer $id
     * @param array $args
     *
     * @return array
     */
    public static function getSettings($id, $args = [])
    {
        $defaults = ['public' => false];
        $args = array_merge($defaults, $args);
        $output = [];

        foreach (self::$settingsModel as $settingModelKey => $settingModel) {
            if ($args['public'] && (!isset($settingModel['public']) || !$settingModel['public'])) {
                continue;
            }

            $output[$settingModelKey] = self::getSetting($id, $settingModelKey);
        }

        return $output;
    }

    /**
     * Get questions list setting for any source type
     *
     * @param integer $id
     *
     * @return array
     */
    public static function getQuestionsList($id)
    {
        $output = Question::getList($id, self::getSetting($id, 'questions'));

        return apply_filters('wcsf_questions_list', $output, $id);
    }
    // </editor-fold>

    // <editor-fold desc="Steps">
    /**
     * Get filter steps array
     *
     * @param integer $id
     *
     * @return array
     */
    public static function getSteps($id)
    {
        static $stepsCache = [];

        if (isset($stepsCache[$id])) {
            return apply_filters('wcsf_steps', $stepsCache[$id], $id);
        }

        $steps = Question::getList($id, self::getSetting($id, 'questions'));
        $steps = array_replace(
            [
                'description' => (object) [
                    'id' => 'description',
                    'name' => get_the_title($id),
                    'description' => get_post_field('post_content', $id)
                ]
            ],
            $steps
        );

        if (self::getSetting($id, 'show_results_within')) {
            $steps['results'] = (object) [
                'id' => 'results',
                'name' => self::getSetting($id, 'results_tab_title')
            ];
        }

        $stepsCache[$id] = $steps;

        return apply_filters('wcsf_steps', $steps, $id);
    }

    /**
     * Get filter steps ids array
     *
     * @param integer $id
     *
     * @return array
     */
    public static function getStepsIds($id)
    {
        $stepsIds = [];

        foreach (self::getSteps($id) as $step) {
            $stepsIds[] = $step->id;
        }

        return apply_filters('wcsf_steps_ids', $stepsIds, $id);
    }

    /**
     * Get filter active step id
     *
     * @param integer $id
     *
     * @return integer
     */
    public static function getActiveStepId($id)
    {
        $activeStep = Storage::get(self::$activeStepsSessionKey, $id);

        if (!$activeStep) {
            $stepsIds = self::getStepsIds($id);
            $activeStep = reset($stepsIds);
        }

        return $activeStep;
    }

    /**
     * Get filter active step
     *
     * @param integer $id
     *
     * @return object|null
     */
    public static function getActiveStep($id)
    {
        $steps = self::getSteps($id);
        $activeStepId = self::getActiveStepId($id);
        $output = null;

        foreach ($steps as $step) {
            if ($step->id == $activeStepId) {
                $output = $step;

                break;
            }
        }

        return apply_filters('wcsf_active_step', $output, $id);
    }

    /**
     * Get filter the next active step
     *
     * @param integer $id
     *
     * @return integer|string
     */
    public static function getNextActiveStepId($id)
    {
        $stepsIds = self::getStepsIds($id);
        $activeStepId = self::getActiveStepId($id);
        $foundActive = false;

        foreach ($stepsIds as $stepId) {
            if ($foundActive) {
                return $stepId;

                break;
            }

            if ($stepId == $activeStepId) {
                $foundActive = true;
            }
        }

        return null;
    }

    /**
     * Get filter the prev active step
     *
     * @param integer $id
     *
     * @return integer|string
     */
    public static function getPrevActiveStepId($id)
    {
        $stepsIds = self::getStepsIds($id);
        $activeStepId = self::getActiveStepId($id);
        $prevStep = null;

        foreach ($stepsIds as $stepId) {
            if ($stepId == $activeStepId) {
                return $prevStep;
            }

            $prevStep = $stepId;
        }

        return null;
    }

    /**
     * Set active filter step to the session variable
     *
     * @param integer $id
     * @param integer|string $step
     */
    public static function setActiveStep($id, $step)
    {
        Storage::set(self::$activeStepsSessionKey, $id, $step);
    }

    /**
     * Reset filter values and set the first step as active
     *
     * @param integer $id
     * @param integer $questionId
     */
    public static function resetState($id, $questionId = null)
    {
        self::resetValue($id, $questionId);

        if (!$questionId) {
            $stepsIds = self::getStepsIds($id);

            self::setActiveStep($id, $stepsIds[0]);
        }
    }
    // </editor-fold>

    // <editor-fold desc="Notices">
    /**
     * Add notice by type into a queue
     *
     * @param string $stepId
     * @param array $massageData
     */
    public function addNotice($stepId, $massageData)
    {
        $this->notices[$stepId][] = $massageData;
    }

    /**
     * Return the queue of notices
     *
     * @param string $stepId - try to get messages from one step by id or output all of messages
     *
     * @return array
     */
    public function getNotices($stepId = null)
    {
        if ($stepId) {
            // return step's messages array or nothing
            return isset($this->notices[$stepId]) ? $this->notices[$stepId] : [];
        } else {
            // return all steps messages
            return array_reduce($this->notices, 'array_merge', []);
        }
    }
    // </editor-fold>

    // <editor-fold desc="Value">
    /**
     * Save the filter value
     * Value should be an array like [stepId => [value1 => 123, value2 = 345]]
     *
     * @param integer $id
     * @param array $value
     *
     * @throws \Exception
     */
    public static function setValue($id, $value)
    {
        foreach ((array) $value as $questionId => $valueItem) {
            foreach ($valueItem as $stepValueKey => $stepValue) {
                // clear empty items, but not zero
                if ($stepValue == '') {
                    unset($valueItem[$stepValueKey]);
                }
            }

            Storage::set(self::$valuesSessionKey, $id, $valueItem, $questionId);
        }
    }

    /**
     * Reset the filter by step id or fully
     *
     * @param integer $id
     * @param integer $questionId
     */
    public static function resetValue($id, $questionId = null)
    {
        if ($questionId) {
            Storage::set(self::$valuesSessionKey, $id, [], $questionId);
        } else {
            Storage::set(self::$valuesSessionKey, $id, []);
        }
    }

    /**
     * Get the filter step value
     *
     * @param integer $id
     *
     * @return array
     */
    public static function getValue($id)
    {
        $output = (array) Storage::get(self::$valuesSessionKey, $id);
        $output = self::removeBorderlineNumberValues($id, $output);

        return apply_filters('wcsf_value', array_filter($output), $id);
    }

    /**
     * Remove number view values which equals to their min/max values
     *
     * @param integer $id
     * @param array $values
     *
     * @return array
     */
    public static function removeBorderlineNumberValues($id, $values)
    {
        foreach (array_keys($values) as $questionId) {
            $question = Question::get($questionId);
            $minMax = [];

            if (!is_object($question) || !property_exists($question, 'viewType')
                || strpos($question->viewType, 'number') === false
            ) {
                continue;
            }

            // get min/max values
            switch ($question->filterType) {
                case 'price':
                    $minMax = Utils::getMinMaxPrices();
                    break;

                case 'attribute':
                    $minMax = Utils::getMinMaxAttributeValues($question->attribute);
                    break;

                case 'meta':
                    $minMax = Utils::getMinMaxMetaValues($question->metaKey);
            }

            switch ($question->viewType) {
                case 'number-between':
                    if (isset($values[$questionId]['from']) && $values[$questionId]['from'] == $minMax['min']
                        && isset($values[$questionId]['to']) && $values[$questionId]['to'] == $minMax['max']
                    ) {
                        unset($values[$questionId]);
                    }
            }
        }

        return apply_filters('wcsf_filtered_borderline_number_values', $values, $id);
    }
    
    /**
     * Get the filter step value list
     *
     * @param integer $id
     *
     * @return array
     */
    public static function getValueList($id)
    {
        $value = self::getValue($id);
        $stepsIds = self::getStepsIds($id);
        $output = [];

        // sorting the values according the steps
        $value = array_replace(array_flip($stepsIds), $value);

        foreach ($value as $questionId => $valueItem) {
            if (!is_array($valueItem)) {
                continue;
            }

            $values = [];
            $question = Question::get($questionId);
            $anyProductValueText = 'Doesn\'t matter';
            $listItem = (object) [
                'id' => $questionId,
                'name' => wp_kses_post($question->name),
                'value' => null,
                'values' => []
            ];

            if (strpos($question->viewType, 'number') !== false) {
                $unit = Question::getSetting($questionId, 'numeric_unit');
                $fromText = Question::getSetting($questionId, 'from_value_text');
                $toText = Question::getSetting($questionId, 'to_value_text');

                // get numeric values
                if ($question->viewType == 'number' && !empty($valueItem)) {
                    $values[] = reset($valueItem) . $unit;
                }

                if (isset($valueItem['from']) && $question->viewType == 'number-between') {
                    $values[] = "$fromText $valueItem[from]$unit";
                }

                if (isset($valueItem['to']) && $question->viewType == 'number-between') {
                    $values[] = "$toText $valueItem[to]$unit";
                }

                $listItem->value = implode(' ', $values);
                $listItem->values = $values;
            } else {
                if ($question->filterType == 'manual') {
                    // get manual names
                    foreach ($valueItem as $valueItemPart) {
                        if (isset($question->manualValues[$valueItemPart])) {
                            $values[] = $question->manualValues[$valueItemPart]['name'];
                        }
                    }
                } elseif (in_array($question->filterType, ['tag', 'category', 'attribute'])) {
                    // define taxonomy
                    switch ($question->filterType) {
                        default:
                        case 'category':
                            $taxonomy = 'product_cat';
                            break;

                        case 'tag':
                            $taxonomy = 'product_tag';
                            break;

                        case 'attribute':
                            $taxonomy = \wc_attribute_taxonomy_name_by_id((int) $question->attribute);
                            break;
                    }

                    // get terms values
                    foreach ($valueItem as $valueItemKey => $valueItemPart) {
                        // get multi-choice term name
                        if (strpos($question->viewType, 'multi-choice') !== false) {
                            if (in_array((int) $valueItemPart, [-1, 1, 2])) {
                                $term = get_term($valueItemKey, $taxonomy);

                                switch ((int) $valueItemPart) {
                                    case -1:
                                        $values[] = "<s class=\"wcsf-value-item-value-excluded\">$term->name</s>";
                                        break;

                                    case 1:
                                        $values[] = "<span class=\"wcsf-value-item-value-possible\">$term->name</span>";
                                        break;

                                    case 2:
                                    default:
                                        $values[] = "<b class=\"wcsf-value-item-value-required\">$term->name</b>";
                                }
                            }

                            continue;
                        }

                        // get "Doesn't matter" text
                        if ($valueItemPart == 0) {
                            $values[] = $anyProductValueText;

                            continue;
                        }

                        // get term name
                        $term = get_term($valueItemPart, $taxonomy);
                        $values[] = $term->name;
                    }
                } elseif ($question->filterType == 'meta') {
                    foreach ($valueItem as $valueItemKey => $valueItemPart) {
                        // get multi-choice term name
                        if (strpos($question->viewType, 'multi-choice') !== false) {
                            if (in_array((int) $valueItemPart, [-1, 1, 2])) {
                                switch ((int) $valueItemPart) {
                                    case -1:
                                        $values[] = "<s class=\"wcsf-value-item-value-excluded\">$valueItemKey</s>";
                                        break;

                                    case 1:
                                        $values[] = "<span class=\"wcsf-value-item-value-possible\">$valueItemKey</span>";
                                        break;

                                    case 2:
                                    default:
                                        $values[] = "<b class=\"wcsf-value-item-value-required\">$valueItemKey</b>";
                                }
                            }

                            continue;
                        }

                        // get "Doesn't matter" text
                        if ($valueItemPart == '0') {
                            $values[] = $anyProductValueText;

                            continue;
                        }

                        $values[] = $valueItemPart;
                    }
                }

                $listItem->value = implode(', ', $values);
                $listItem->values = $values;
            }

            if (!empty($listItem->value)) {
                $output[] = $listItem;
            }
        }

        return apply_filters('wcsf_value_list', $output, $id);
    }
    // </editor-fold>

    // <editor-fold desc="Nav">
    /**
     * Get filter navigation items array
     *
     * @param integer $id
     *
     * @return array
     */
    public static function getNavItems($id)
    {
        $steps = self::getSteps($id);
        $activeStep = self::getActiveStepId($id);
        $output = [];
        $metActiveItem = false;

        foreach ($steps as $step) {
            $isActiveStep = $activeStep == $step->id;
            $navItem = [
                'id' => $step->id,
                'name' => $step->name,
                'class' => '',
                'component' => 'wcsf-submit-and-go-to',
                'type' => 'submit',
                'state' => 'default',
                'disabled' => false
            ];

            if ($isActiveStep) {
                $metActiveItem = true;
            }

            $navItem['type'] = $isActiveStep ? 'button' : 'submit';
            $navItem['state'] = $isActiveStep ? 'active' : 'default';
            $navItem['class'] = $isActiveStep ? 'active' : ($metActiveItem ? 'default' : 'past');
            $navItem['component'] = $isActiveStep ? 'wcsf-none' : 'wcsf-submit-and-go-to';

            $output[] = $navItem;
        }

        return apply_filters('wcsf_nav_items', $output, $id);
    }

    /**
     * Get pagination items array for products loop
     *
     * @param array $args
     *
     * @return array
     */
    public static function getProductsPaginationItems($args)
    {
        $output = [];
        $defaults = [
            'page' => 1,
            'productsQuery' => []
        ];

        $args = array_replace_recursive($defaults, $args);

        if (!$args['productsQuery'] || empty($args['productsQuery'])) {
            return [];
        }

        $paginationArgs = apply_filters(
            'wcsf_products_pagination_args',
            [
                'format' => '?paged=%#%',
                'base' => '%_%',
                'total' => $args['productsQuery']->max_num_pages,
                'current' => max($args['page'], get_query_var('paged')),
                'show_all' => false,
                'end_size' => 1,
                'mid_size' => 2,
                'prev_next' => true,
                'prev_text'    => '&larr;',
                'next_text'    => '&rarr;',
                'type' => 'array'
            ],
            $args
        );

        $links = paginate_links($paginationArgs);

        foreach ((array) $links as $link) {
            // add custom classes
            $link = str_replace('page-numbers', 'page-numbers page-link', $link);

            // replace empty href
            $link = str_replace('href=""', 'href="?paged=1"', $link);
            $link = str_replace("href=''", 'href="?paged=1"', $link);

            preg_match_all('/<a[^>]+href=([\'"])(?<href>.+?)\1[^>]*>/i', $link, $result);

            if (!empty($result) && !empty($result['href'][0])) {
                $href = $result['href'][0];
                $linkParts = parse_url($href);

                parse_str($linkParts['query'], $linkPartsQuery);

                $paged = isset($linkPartsQuery['paged']) ? $linkPartsQuery['paged'] : 1;

                // add custom attributes
                $link = str_replace(
                    ' href=',
                    ' data-component="wcsf-products-pagination-link wcsf-get-step" data-step-id="results" data-page="'
                    . $paged . '" href=',
                    $link
                );
            }

            $output[] = (object) [
                'class' => strpos($link, 'current') !== false ? 'active' : '',
                'innerHtml' => $link
            ];
        }

        return apply_filters('wcsf_products_pagination_items', $output, $args);
    }
    // </editor-fold>
    
    // <editor-fold desc="Results">
    /**
     * Get filter results URL with query args
     *
     * @param integer $id
     * @param array $queryArgs - extra query args
     *
     * @return string
     */
    public static function getResultsUrl($id, $queryArgs = [])
    {
        $url = self::getSetting($id, 'results_url');
        $query = [];

        // if the settings is empty
        if (!$url) {
            $url = get_permalink(wc_get_page_id('shop'));
        }

        // will stay on the same page
        if ($url == '#') {
            if (isset($queryArgs['wcsf-referer'])) {
                $url = $queryArgs['wcsf-referer'];
            } else {
                $url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] == 'on' ? 'https' : 'http')
                    . "://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";
            }
        }

        // if URL is absolute
        if (strpos($url, $_SERVER['HTTP_HOST']) === false) {
            $url = home_url() . '/' . $url;
        }

        $parts = parse_url($url);

        // parse query args
        if (isset($parts['query'])) {
            parse_str($parts['query'], $query);
        }

        // redirect to the product archive
        if (!empty($query['page_id']) && '' === get_option('permalink_structure')
            && wc_get_page_id('shop') == $query['page_id']
        ) {
            $url = get_post_type_archive_link('product');
            $parts = parse_url($url);
            $query = [];

            // parse query args
            if (isset($parts['query'])) {
                parse_str($parts['query'], $query);
            }
        }

        // blend in required args
        $query['wcsf-results'] = $id;
        $query['wcsf-value'] = self::getValue($id);
        $query['wcsf-referer'] = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] == 'on' ? 'https' : 'http')
            . "://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";

        $query = array_replace($query, $queryArgs);

        // unset reset query
        if (isset($query['wcsf-reset-value'])) {
            unset($query['wcsf-reset-value']);
        }

        if (isset($query['wcsf-reset-state'])) {
            unset($query['wcsf-reset-state']);
        }

        // woof plugin query emulate
        if (class_exists('\\WOOF')) {
            $query['swoof'] = 1;
        }

        // apply outer filters
        $query = apply_filters('wcsf_results_redirect_url_query', $query, $id);

        // build query string
        $parts['query'] = http_build_query($query);

        return apply_filters('wcsf_results_redirect_url', Utils::buildUrl($parts), $id, $parts);
    }

    /**
     * Get filter reset URL
     *
     * @param integer $id
     * @param array $queryArgs - extra query args
     *
     * @return string
     */
    public static function getResetUrl($id, $queryArgs = [])
    {
        $query = [];

        if (isset($queryArgs['wcsf-referer'])) {
            $url = $queryArgs['wcsf-referer'];
        } else {
            $url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] == 'on' ? 'https' : 'http')
                . "://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";
        }

        // if the settings is empty
        if (!$url) {
            $url = get_permalink(wc_get_page_id('shop'));
        }

        // if URL is absolute
        if (strpos($url, $_SERVER['HTTP_HOST']) === false) {
            $url = home_url() . '/' . $url;
        }

        $parts = parse_url($url);

        // parse query args
        if (isset($parts['query'])) {
            parse_str($parts['query'], $query);
        }

        // add reset state
        $query['wcsf-reset-state'] = $id;

        $query = array_replace($query, $queryArgs);

        // clear the query
        unset($query['wcsf-reset-value']);
        unset($query['wcsf-results']);
        unset($query['wcsf-value']);
        unset($query['wcsf-referer']);

        // apply outer filters
        $query = apply_filters('wcsf_results_reset_url_query', $query, $id);

        // build query string
        $parts['query'] = http_build_query($query);

        return apply_filters('wcsf_reset_redirect_url', Utils::buildUrl($parts), $id, $parts);
    }

    /**
     * Get pagination items array for products loop
     *
     * @param integer $id
     * @param array $args
     *
     * @return array
     */
    public static function getProductsQueryArgs($id, $args = [])
    {
        $defaults = [
            'orderBy' => 'menu_order',
            'page' => 1
        ];

        $args = array_replace($defaults, $args);
        $orderParts = explode('-', $args['orderBy']);
        $order = isset($orderParts[1]) ? $orderParts[1] : 'asc';
        $value = self::getValue($id);
        $output = Core::getQueryArgsByFilter($id, $value, $args);
        $wcQuery = new \WC_Query();
        $orderingArgs = $wcQuery->get_catalog_ordering_args($orderParts[0], $order);
        $defaults = [
            'post_type' => 'product',
            'posts_per_page' => wc_get_default_products_per_row() * wc_get_default_product_rows_per_page(),
            'orderby' => $args['orderBy'],
            'paged' => $args['page'],
            'post_status' => 'publish',
            'tax_query' => []
        ];

        if (filter_var(get_option('woocommerce_hide_out_of_stock_items'), FILTER_VALIDATE_BOOLEAN)) {
            $defaults['tax_query'][] = [
                'taxonomy' => 'product_visibility',
                'field' => 'name',
                'terms' => 'outofstock',
                'operator' => 'NOT IN'
            ];
        }

        $output = array_replace_recursive($defaults, $args, $output, $orderingArgs);

        return apply_filters('wcsf_products_query_args', $output, $id, $args);
    }
    // </editor-fold>

    // <editor-fold desc="Get">
    /**
     * Returns array of ids and titles of posts
     *
     * @param array $args
     *
     * @return array
     */
    public static function getPostsIdsList($args = [])
    {
        $defaults = [
            'post_type' => 'wcsf',
            'numberposts' => -1,
            'post_status' => 'publish'
        ];

        $args = array_replace($defaults, $args);
        $output = [];
        $posts = get_posts($args);

        foreach ($posts as $post) {
            $output[$post->ID] = $post->post_title;
        }

        return $output;
    }
    // </editor-fold>

    // <editor-fold desc="AJAX">
    /** Save value and get a step form template by ajax */
    public function submitAndGoToAjax()
    {
        $postData = Utils::parseArrayOfJSONs($_POST);
        $defaults = [
            'id' => null,
            'stepId' => null,
            'page' => 1,
            'orderBy' => 'menu_order',
            'referer' => home_url(),
            'value' => []
        ];

        $args = array_replace($defaults, $postData);

        try {
            // save value
            self::setValue($args['id'], $args['value']);
        } catch (\Exception $exception) {
            $this->addNotice(
                $exception->getCode() ? $exception->getCode() : self::getActiveStepId($postData['id']),
                ['message' => $exception->getMessage()]
            );

            Utils::sendJSON([
                'error' => true,
                'message' => $exception->getMessage(),
                'content' => Core::getTemplatePart(
                    'form',
                    [
                        'id' => $args['id'],
                        'page' => $args['page'],
                        'orderBy' => $args['orderBy'],
                        'resetUrl' => self::getResetUrl($args['id'], ['wcsf-referer' => $args['referer']])
                    ],
                    ['echo' => false]
                )
            ]);
        }

        // change step
        if ($args['stepId']) {
            self::setActiveStep($args['id'], $args['stepId']);
        }

        Utils::sendJSON([
            'content' => Core::getTemplatePart(
                'form',
                [
                    'id' => $args['id'],
                    'page' => $args['page'],
                    'orderBy' => $args['orderBy'],
                    'resetUrl' => self::getResetUrl($args['id'], ['wcsf-referer' => $args['referer']])
                ],
                ['echo' => false]
            )
        ]);
    }

    /** Save value and get the next form template by ajax */
    public function submitAndGoNextAjax()
    {
        $postData = Utils::parseArrayOfJSONs($_POST);
        $defaults = [
            'id' => null,
            'stepId' => null,
            'page' => 1,
            'orderBy' => 'menu_order',
            'referer' => home_url(),
            'value' => []
        ];

        $args = array_replace($defaults, $postData);

        try {
            // save value
            self::setValue($args['id'], $args['value']);
        } catch (\Exception $exception) {
            $this->addNotice(
                $exception->getCode() ? $exception->getCode() : self::getActiveStepId($postData['id']),
                ['message' => $exception->getMessage()]
            );

            Utils::sendJSON([
                'error' => true,
                'message' => $exception->getMessage(),
                'content' => Core::getTemplatePart(
                    'form',
                    [
                        'id' => $args['id'],
                        'page' => $args['page'],
                        'orderBy' => $args['orderBy'],
                        'resetUrl' => self::getResetUrl($args['id'], ['wcsf-referer' => $args['referer']])
                    ],
                    ['echo' => false]
                )
            ]);
        }

        // get steps after value update
        $stepsIds = self::getStepsIds($args['id']);

        if (end($stepsIds) == $args['stepId']) {
            Utils::sendJSON(['redirect' => self::getResultsUrl($args['id'], ['wcsf-referer' => $args['referer']])]);
        }

        // set active step from the step id and increment it
        self::setActiveStep($args['id'], $args['stepId']);
        self::setActiveStep($args['id'], self::getNextActiveStepId($args['id']));

        Utils::sendJSON([
            'content' => Core::getTemplatePart(
                'form',
                [
                    'id' => $args['id'],
                    'page' => $args['page'],
                    'orderBy' => $args['orderBy'],
                    'resetUrl' => self::getResetUrl($args['id'], ['wcsf-referer' => $args['referer']])
                ],
                ['echo' => false]
            )
        ]);
    }

    /** Save value and get the form template by ajax */
    public function getStepHtmlAjax()
    {
        $postData = Utils::parseArrayOfJSONs($_POST);
        $defaults = [
            'id' => null,
            'stepId' => null,
            'page' => 1,
            'value' => null,
            'orderBy' => 'menu_order',
            'referer' => home_url(),
        ];

        $args = array_replace($defaults, $postData);

        // change step
        if ($args['stepId']) {
            self::setActiveStep($args['id'], $args['stepId']);
        }

        Utils::sendJSON([
            'content' => Core::getTemplatePart(
                'form',
                [
                    'id' => $args['id'],
                    'stepId' => $args['stepId'],
                    'page' => $args['page'],
                    'orderBy' => $args['orderBy'],
                    'resetUrl' => self::getResetUrl($args['id'], ['wcsf-referer' => $args['referer']])
                ],
                ['echo' => false]
            )
        ]);
    }

    /** Reset filter by ajax */
    public function resetAjax()
    {
        $postData = Utils::parseArrayOfJSONs($_POST);
        $defaults = [
            'id' => null,
            'resetStepId' => null
        ];

        $args = array_replace($defaults, $postData);

        if (!$args['id']) {
            return;
        }

        self::resetState($args['id'], $args['resetStepId']);

        Utils::sendJSON(['content' => Core::getTemplatePart('form', ['id' => $args['id']], ['echo' => false])]);
    }
    // </editor-fold>
}
