<?php
namespace WCStepFilter;

/**
 * Question Class
 *
 * @class Question
 * @version 7.0.0
 */
class Question
{
    // <editor-fold desc="Properties">
    /**
     * Settings model variable
     *
     * @var array
     */
    public static $settingsModel;
    // </editor-fold>

    // <editor-fold desc="Core">
    /** Constructor */
    public function __construct()
    {
        add_action('init', [$this, 'onInit']);

        // admin scripts enqueue
        add_action('admin_enqueue_scripts', [$this, 'enqueueAdmin']);

        // fields
        add_action('add_meta_boxes', [$this, 'addMetaBoxes']);
        add_action('save_post_wcsf_question', [$this, 'savePostAction']);

        // ajax
        add_action('wp_ajax_wcsfAddQuestionValue', [$this, 'addQuestionValueAjax']);
        add_action('wp_ajax_wcsfGetQuestionValuesSelectOptions', [$this, 'getValuesSelectOptionsAjax']);
    }

    /** On init action */
    public function onInit()
    {
        self::$settingsModel = apply_filters(
            'wcsf_question_settings_model',
            [
                'view_type' => [
                    'label' => L10N::r('View type'),
                    'key' => '_view_type',
                    'type' => 'select',
                    'default' => 'select',
                    'values' => [
                        'select' => L10N::r('Select'),
                        'checkbox' => L10N::r('Checkbox'),
                        'number' => L10N::r('Number'),
                        'number-between' => L10N::r('Number between'),
                        'multi-choice-range' => L10N::r('Multi-choice range')
                    ]
                ],
                'filter_type' => [
                    'label' => L10N::r('Filter type'),
                    'key' => '_filter_type',
                    'type' => 'select',
                    'default' => 'select',
                    'values' => [
                        'attribute' => L10N::r('Attribute'),
                        'price' => L10N::r('Price'),
                        'tag' => L10N::r('Tag'),
                        'category' => L10N::r('Category'),
                        'manual' => L10N::r('Manual'),
                        'meta' => L10N::r('Meta')
                    ]
                ],
                'meta_key' => [
                    'label' => L10N::r('Meta key'),
                    'key' => '_meta_key',
                    'type' => 'text',
                    'default' => ''
                ],
                'multi_choice_undefined_value_text' => [
                    'label' => L10N::r('"Undefined" value text'),
                    'key' => '_multi_choice_undefined_value_text',
                    'type' => 'text',
                    'default' => L10N::r('Undefined')
                ],
                'multi_choice_possible_value_text' => [
                    'label' => L10N::r('"Possible" value text'),
                    'key' => '_multi_choice_possible_value_text',
                    'type' => 'text',
                    'default' => L10N::r('Possible')
                ],
                'multi_choice_required_value_text' => [
                    'label' => L10N::r('"Required" value text'),
                    'key' => '_multi_choice_required_value_text',
                    'type' => 'text',
                    'default' => L10N::r('Required')
                ],
                'multi_choice_excluded_value_text' => [
                    'label' => L10N::r('"Excluded" value text'),
                    'key' => '_multi_choice_excluded_value_text',
                    'type' => 'text',
                    'default' => L10N::r('Excluded')
                ],
                'from_value_text' => [
                    'label' => L10N::r('"From" value text'),
                    'key' => '_from_value_text',
                    'type' => 'text',
                    'default' => L10N::r('From')
                ],
                'to_value_text' => [
                    'label' => L10N::r('"To" value text'),
                    'key' => '_to_value_text',
                    'type' => 'text',
                    'default' => L10N::r('To')
                ],
                'numeric_unit' => [
                    'label' => L10N::r('Numeric unit'),
                    'key' => '_numeric_unit',
                    'type' => 'text',
                    'default' => '$'
                ],
                'category_parent' => [
                    'label' => L10N::r('Category parent'),
                    'key' => '_category_parent',
                    'type' => 'select',
                    'default' => 0,
                    'values' => []
                ],
                'hide_empty_terms' => [
                    'label' => L10N::r('Hide empty terms'),
                    'key' => '_hide_empty_terms',
                    'type' => 'checkbox',
                    'default' => false
                ],
                'show_count' => [
                    'label' => L10N::r('Show count'),
                    'key' => '_show_count',
                    'type' => 'checkbox',
                    'default' => false,
                    'description' => L10N::r('Show term products count')
                ],
                'attribute' => [
                    'label' => L10N::r('Attribute'),
                    'key' => '_attribute',
                    'type' => 'select',
                    'default' => 0,
                    'values' => []
                ],
                'relation' => [
                    'label' => L10N::r('Relation'),
                    'key' => '_relation',
                    'type' => 'select',
                    'default' => 'or',
                    'values' => [
                        'or' => L10N::r('OR'),
                        'and' => L10N::r('AND')
                    ]
                ],
                'manual_values' => [
                    'label' => L10N::r('Manual values'),
                    'key' => '_manual_values',
                    'type' => 'data-table',
                    'default' => []
                ]
            ]
        );

        $name = 'SF Questions';
        $args = [
            'label' => $name,
            'labels' => [
                'name' => $name,
                'singular_name' => $name,
                'menu_name' => $name
            ],
            'description' => L10N::r('This is where you can add new questions for step filter'),
            'public' => false,
            'show_ui' => true,
            'map_meta_cap' => true,
            'publicly_queryable' => false,
            'exclude_from_search' => true,
            'show_in_menu' => current_user_can('manage_woocommerce') ? 'woocommerce' : true,
            'hierarchical' => false,
            'rewrite' => false,
            'query_var' => false,
            'supports' => [
                'title',
                'editor',
                'page-attributes'
            ],
            'show_in_nav_menus' => false,
            'show_in_admin_bar' => true
        ];

        register_post_type('wcsf_question', $args);
    }
    // </editor-fold>

    // <editor-fold desc="Admin">
    /** Admin styles and scripts enqueue */
    public function enqueueAdmin()
    {
        wp_enqueue_script(
            'woocommerce-step-filter-question',
            WC_STEP_FILTER_PLUGIN_URL . 'assets/admin/js/question.min.js',
            ['jquery'],
            WC_STEP_FILTER_VERSION,
            true
        );

        if (defined('WC_VERSION')
            && ((isset($_GET['post']) && get_post_type((int) $_GET['post']) == 'wcsf_question')
                || (isset($_GET['post_type']) && $_GET['post_type'] == 'wcsf_question'))
        ) {
            wp_register_script(
                'select2',
                \WC()->plugin_url() . '/assets/js/select2/select2.full.min.js',
                ['jquery'],
                '4.0.3'
            );

            wp_register_script(
                'selectWoo',
                WC()->plugin_url() . '/assets/js/selectWoo/selectWoo.full.min.js',
                ['jquery'],
                '1.0.0'
            );

            wp_register_script(
                'wc-enhanced-select',
                WC()->plugin_url() . '/assets/js/admin/wc-enhanced-select.min.js',
                ['jquery', 'selectWoo'],
                WC_VERSION
            );

            wp_enqueue_script('select2');
            wp_enqueue_script('selectWoo');
            wp_enqueue_script('wc-enhanced-select');
            wp_enqueue_style('woocommerce_admin_styles', \WC()->plugin_url() . '/assets/css/admin.css', [], WC_VERSION);
        }
    }

    /**
     * Save meta values
     *
     * @param integer $postId
     */
    public function savePostAction($postId)
    {
        if ((defined('DOING_AUTOSAVE') && DOING_AUTOSAVE)
            || !current_user_can('edit_page', $postId)
            || (isset($_REQUEST['action']) && $_REQUEST['action'] == 'inline-save')
        ) {
            return;
        }

        foreach (self::$settingsModel as $setting) {
            $value = '';

            if (isset($_POST[$setting['key']])) {
                $value = $_POST[$setting['key']];
            }

            update_post_meta($postId, $setting['key'], $value);
        }
    }

    /** Add meta-boxes */
    public function addMetaBoxes()
    {
        add_meta_box(
            'woocommerce-step-filter',
            L10N::r('Settings'),
            [$this, 'metaBoxCallback'],
            'wcsf_question',
            'normal'
        );
    }

    /**
     * Question page meta-box content
     *
     * @param object $post
     */
    public function metaBoxCallback($post)
    {
        $manualValues = self::getSetting($post->ID, 'manual_values');
        $attributes = wc_get_attribute_taxonomies();
        $settingsModel = self::$settingsModel;

        // unset settings model for custom output
        unset($settingsModel['manual_values']);

        $settingsModel['category_parent']['values'] = array_replace(
            [0 => ''],
            Utils::getTermsTree(
                ['taxonomy' => 'product_cat'],
                [
                    'flatten' => true,
                    'showLevelInName' => true,
                    'nameOnly' => true
                ]
            )
        );

        foreach ($attributes as $attribute) {
            $settingsModel['attribute']['values'][$attribute->attribute_id] = $attribute->attribute_label;
        }

        $attributeKeys = array_keys($settingsModel['attribute']['values']);
        $settingsModel['attribute']['default'] = reset($attributeKeys);
        ?>
        <div class="wcsf-question"
            data-component="wcsf-question"
            data-post-id="<?php echo esc_attr($post->ID); ?>"
            data-ajax-url="<?php echo admin_url('admin-ajax.php'); ?>">
            <table class="wcsf-settings-table form-table wcsf-data-table">
                <?php foreach ($settingsModel as $settingKey => $setting) { ?>
                    <tr valign="top" data-component="wcsf-setting" data-key="<?php
                    echo esc_attr($settingKey);
                    ?>">
                        <th scope="row" class="titledesc">
                            <label for="<?php echo esc_attr($setting['key']); ?>"><?php
                                echo wp_kses_post($setting['label']);
                                ?></label>
                        </th>
                        <td>
                            <?php
                            Core::settingFieldView(
                                $setting,
                                ['values' => [$setting['key'] => self::getSetting($post->ID, $settingKey)]]
                            );
                            ?>
                        </td>
                    </tr>
                <?php } ?>
                <tr class="wcsf-question-manual" data-component="wcsf-setting" data-key="manual">
                    <th><?php esc_html_e('Manual values', 'woocommerce-step-filter'); ?></th>
                    <td>
                        <button class="button-primary"
                            data-component="wcsf-question-add-value"><?php
                            esc_html_e('Add a value', 'woocommerce-step-filter');
                            ?></button><br><br>
                        <div class="wcsf-question-manual-values">
                            <table class="wp-list-table widefat striped">
                                <thead>
                                    <tr>
                                        <th><?php esc_html_e('Pre-checked', 'woocommerce-step-filter'); ?></th>
                                        <th><?php esc_html_e('Name', 'woocommerce-step-filter'); ?></th>
                                        <th><?php esc_html_e('Source', 'woocommerce-step-filter'); ?></th>
                                        <th><?php esc_html_e('Remove', 'woocommerce-step-filter'); ?></th>
                                    </tr>
                                </thead>
                                <tbody data-component="wcsf-question-manual-values">
                                    <?php
                                    if (is_array($manualValues)) {
                                        foreach ($manualValues as $valueKey => $value) {
                                            $this->renderManualQuestionValue(
                                                array_merge(
                                                    (array) $value,
                                                    [
                                                        'postId' => $post->ID,
                                                        'index' => $valueKey
                                                    ]
                                                )
                                            );
                                        }
                                    } else {
                                        $this->renderManualQuestionValue(['postId' => $post->ID]);
                                    }
                                    ?>
                                </tbody>
                            </table>
                        </div>
                    </td>
                </tr>
            </table>
        </div>
        <?php
    }

    /** Return manual value html */
    public function addQuestionValueAjax()
    {
        $this->renderManualQuestionValue($_GET);

        exit();
    }

    /**
     * Return table of manual values html
     *
     * @param array $args
     */
    public function renderManualQuestionValue($args = [])
    {
        $defaults = [
            'postId' => 0,
            'index' => 0,
            'name' => '',
            'pre-checked' => 0,
            'filter-by' => 'any'
        ];

        $args = array_replace($defaults, $args);
        ?>
        <tr data-component="wcsf-question-manual-value" data-index="<?php echo esc_attr($args['index']); ?>">
            <td>
                <input type="hidden"
                    name="_manual_values[<?php echo esc_attr($args['index']); ?>][pre-checked]"
                    value="0">
                <label>
                    <input type="checkbox"
                        name="_manual_values[<?php echo esc_attr($args['index']); ?>][pre-checked]"
                        class="regular-text wcsf-question-manual-values-item-pre-checked"
                        value="1"
                        <?php echo $args['pre-checked'] ? ' checked="checked"' : ''; ?>>
                    <span class="screen-reader-text"><?php
                        esc_html_e('Pre-checked', 'woocommerce-step-filter');
                        ?></span>
                </label>
            </td>
            <td>
                <input type="text"
                    name="_manual_values[<?php echo esc_attr($args['index']); ?>][name]"
                    class="regular-text wcsf-question-manual-values-item-name"
                    placeholder="<?php esc_attr_e('Some value here', 'woocommerce-step-filter'); ?>"
                    value="<?php echo esc_attr($args['name']); ?>">
            </td>
            <td>
                <a href="#wcsf-data-table-modal-source-<?php echo esc_attr($args['index']); ?>"
                    class="button-primary wcsf-data-table-item-open-modal"
                    data-component="wcsf-open-modal"><?php
                    esc_html_e('Set', 'woocommerce-step-filter');
                    ?></a>
                <div id="wcsf-data-table-modal-source-<?php echo esc_attr($args['index']); ?>"
                    class="wcsf-modal"
                    data-component="wcsf-modal">
                    <div class="wcsf-modal-dialog">
                        <a href="#close"
                            title="<?php esc_html_e('Close', 'woocommerce-step-filter'); ?>"
                            class="wcsf-modal-close"
                            data-component="wcsf-modal-close">&times;</a>
                        <ul>
                            <?php
                            $this->renderAnyProductField($args);
                            $this->renderProductSelectFields($args);
                            $this->renderTagSelectFields($args);
                            ?>
                        </ul>
                        <div class="wcsf-modal-dialog-footer">
                            <a href="#close" class="button button-primary" data-component="wcsf-modal-close"><?php
                                esc_html_e('Save', 'woocommerce-step-filter');
                                ?></a>
                        </div>
                    </div>
                </div>
            </td>
            <td>
                <button class="button-secondary"
                    title="<?php esc_attr_e('Remove', 'woocommerce-step-filter'); ?>"
                    data-component="wcsf-question-manual-value-remove"
                    data-index="<?php echo esc_attr($args['index']); ?>">&times;</button>
            </td>
        </tr>
        <?php
    }

    /**
     * Get list of an option values
     *
     * @param integer|array $id
     *
     * @return array
     */
    public function getValuesSelectOptions($id)
    {
        $output = [];
        $filterType = self::getSetting($id, 'filter_type');
        $viewType = self::getSetting($id, 'view_type');

        if ($filterType == 'manual') {
            foreach ((array) self::getSetting($id, 'manual_values') as $key => $value) {
                $output[$key] = $value['name'];
            }
        } elseif ($filterType == 'attribute') {
            $taxonomy = wc_attribute_taxonomy_name_by_id((int) self::getSetting($id, 'attribute'));
            $terms = get_terms([
                'taxonomy' => $taxonomy,
                'hide_empty' => false
            ]);

            foreach ($terms as $term) {
                $key = strpos($viewType, 'number') !== false ? $term->name : $term->term_id;
                $output[$key] = $term->name;
            }
        } elseif ($filterType == 'price') {
            static $prices;

            if (empty($prices)) {
                $prices = Utils::getProductsPrices();
            }

            foreach ($prices as $price) {
                $output[$price] = $price;
            }
        } elseif ($filterType == 'tag') {
            $terms = Utils::getTermsTree(
                ['taxonomy' => 'product_tag'],
                [
                    'flatten' => true,
                    'showLevelInName' => true
                ]
            );

            foreach ($terms as $term) {
                $output[$term->term_id] = $term->name;
            }
        }

        asort($output, SORT_NUMERIC);

        return $output;
    }

    /** Get list of an option values via Ajax */
    public function getValuesSelectOptionsAjax()
    {
        $output = [];
        $values = $this->getValuesSelectOptions((int) $_GET['value']);

        foreach ($values as $key => $value) {
            $output[] = "<option value=\"$key\">$value</option>";
        }

        echo implode('', $output);

        exit;
    }

    /**
     * Return "any product" radio html
     *
     * @param array $args
     */
    public function renderAnyProductField($args = [])
    {
        $defaults = [
            'index' => 0,
            'filter-by' => 'any'
        ];

        $args = array_replace($defaults, $args);

        $isChecked = $args['filter-by'] == 'any';
        ?>
        <li class="wcsf-question-manual-value-filter<?php echo !$isChecked ? ' is-closed' : ''; ?>"
            data-component="wcsf-question-manual-value-filter"
            data-id="any">
            <label class="wcsf-question-manual-value-filter-label">
                <input type="radio"
                    data-component="wcsf-question-manual-value-filter-by"
                    data-id="any"
                    name="_manual_values[<?php echo esc_attr($args['index']); ?>][filter-by]"
                    value="any"<?php echo $isChecked ? ' checked="checked"' : ''; ?>>
                <?php esc_html_e('Any product', 'woocommerce-step-filter'); ?>
            </label>
        </li>
        <?php
    }

    /**
     * Return filter by tags radio html
     *
     * @param array $args
     */
    public function renderTagSelectFields($args = [])
    {
        $defaults = [
            'index' => 0,
            'items' => [],
            'type' => 'select',
            'filter-by' => 'tag'
        ];

        $args = array_replace($defaults, $args);

        $isChecked = $args['filter-by'] == 'tag';
        ?>
        <li class="wcsf-question-manual-value-filter<?php echo !$isChecked ? ' is-closed' : ''; ?>"
            data-component="wcsf-question-manual-value-filter"
            data-id="tag">
            <label class="wcsf-question-manual-value-filter-label">
                <input type="radio"
                    data-component="wcsf-question-manual-value-filter-by"
                    data-id="tag"
                    name="_manual_values[<?php echo esc_attr($args['index']); ?>][filter-by]"
                    value="tag"<?php echo $isChecked ? ' checked="checked"' : ''; ?>>
                <?php esc_html_e('Filter by tags', 'woocommerce-step-filter'); ?>
            </label>
            <?php $this->renderTermSelectFieldsValues($args, 'product_tag', 0, $isChecked); ?>
        </li>
        <?php
    }

    /**
     * Return list of categories checkbox html
     *
     * @param array $args
     * @param string $tax
     * @param integer $parent
     * @param bool $isChecked
     *
     * @return string
     */
    public function renderTermSelectFieldsValues($args, $tax, $parent, $isChecked)
    {
        $terms = get_terms([
            'taxonomy' => $tax,
            'parent' => $parent,
            'hide_empty' => false
        ]);

        if (empty($terms) || is_wp_error($terms)) {
            return '';
        }
        ?>
        <table class="wp-list-table widefat striped wcsf-question-manual-value-filter-values">
            <?php foreach ($terms as $term) { ?>
                <tr>
                    <td>
                        <label>
                            <input type="checkbox"
                                name="_manual_values[<?php echo esc_attr($args['index']); ?>][items][]"
                                value="<?php echo esc_attr($term->term_id); ?>"<?php
                                echo $isChecked && in_array($term->term_id, $args['items']) ? ' checked' : '';
                            ?>>
                            <?php echo wp_kses_post($term->name); ?>
                            <small><?php echo wp_kses_post($term->count); ?></small>
                        </label>
                        <?php $this->renderTermSelectFieldsValues($args, $tax, $term->term_id, $isChecked); ?>
                    </td>
                </tr>
            <?php } ?>
        </table>
        <?php
        return '';
    }

    /**
     * Return filter by product html
     *
     * @param array $args
     */
    public function renderProductSelectFields($args = [])
    {
        $defaults = [
            'index' => 0,
            'items' => [],
            'type' => 'select',
            'filter-by' => 'product'
        ];

        $args = array_replace($defaults, $args);

        $isChecked = $args['filter-by'] == 'product';
        ?>
        <li class="wcsf-question-manual-value-filter<?php echo !$isChecked ? ' is-closed' : ''; ?>"
            data-component="wcsf-question-manual-value-filter"
            data-id="product">
            <label class="wcsf-question-manual-value-filter-label">
                <input type="radio"
                    data-component="wcsf-question-manual-value-filter-by"
                    data-id="product"
                    name="_manual_values[<?php echo esc_attr($args['index']); ?>][filter-by]"
                    value="product"<?php echo $isChecked ? ' checked="checked"' : ''; ?>>
                <?php esc_html_e('Select products manually', 'woocommerce-step-filter'); ?>
            </label>
            <div class="wcsf-question-manual-value-filter-values">
                <?php
                Core::settingFieldView(
                    [
                        'label' => L10N::r('Products'),
                        'key' => esc_attr("_manual_values[$args[index]][items]"),
                        'type' => 'wc-product-search',
                        'action' => 'woocommerce_json_search_products',
                        'default' => []
                    ],
                    [
                        'values' => [
                            esc_attr("_manual_values[$args[index]][items]") => $args['filter-by'] == 'product'
                                ? $args['items']
                                : []
                        ]
                    ]
                );
                ?>
            </div>
        </li>
        <?php
    }
    // </editor-fold>

    // <editor-fold desc="Settings">
    /**
     * Get one of post settings
     *
     * @param integer $id
     * @param string $setting
     *
     * @return string|boolean|array|float
     */
    public static function getSetting($id, $setting)
    {
        static $questionSettingsCache;

        if (isset($questionSettingsCache[$id], $questionSettingsCache[$id][$setting])) {
            return apply_filters('wcsf_question_setting', $questionSettingsCache[$id][$setting], $id, $setting);
        }

        $value = isset(self::$settingsModel[$setting]['key'])
            ? get_post_meta($id, self::$settingsModel[$setting]['key'], true)
            : '';

        if ($value == '' && isset(self::$settingsModel[$setting]['default'])) {
            $value = self::$settingsModel[$setting]['default'];
        }

        $value = Core::handleSettingType($value, self::$settingsModel[$setting]['type']);

        if (!isset($questionSettingsCache[$id])) {
            $questionSettingsCache[$id] = [];
        }

        $questionSettingsCache[$id][$setting] = $value;

        return apply_filters('wcsf_question_setting', $value, $id, $setting);
    }
    // </editor-fold>

    // <editor-fold desc="Get">
    /**
     * Return question data object
     *
     * @param object $question - WP query post
     *
     * @return object|null
     */
    public static function getData($question)
    {
        if (!$question || !property_exists($question, 'ID')) {
            return apply_filters('wcsf_question_data', null, $question);
        }

        $viewType = self::getSetting($question->ID, 'view_type');
        $multiChoiceValues = [];

        if (strpos($viewType, 'multi-choice') !== false) {
            $multiChoiceValues = [
                0 => [
                    'label' => self::getSetting($question->ID, 'multi_choice_undefined_value_text'),
                    'key' => 'undefined',
                    'value' => 0
                ],
                1 => [
                    'label' => self::getSetting($question->ID, 'multi_choice_possible_value_text'),
                    'key' => 'possible',
                    'value' => 1
                ],
                2 => [
                    'label' => self::getSetting($question->ID, 'multi_choice_required_value_text'),
                    'key' => 'required',
                    'value' => 2
                ],
                -1 => [
                    'label' => self::getSetting($question->ID, 'multi_choice_excluded_value_text'),
                    'key' => 'excluded',
                    'value' => -1
                ]
            ];
        }

        $output = (object) [
            'id' => (int) $question->ID,
            'name' => $question->post_title,
            'description' => $question->post_content,
            'viewType' => $viewType,
            'filterType' => self::getSetting($question->ID, 'filter_type'),
            'attribute' => self::getSetting($question->ID, 'attribute'),
            'manualValues' => self::getSetting($question->ID, 'manual_values'),
            'relation' => self::getSetting($question->ID, 'relation'),
            'metaKey' => self::getSetting($question->ID, 'meta_key'),
            'multiChoiceValues' => array_values($multiChoiceValues)
        ];

        return apply_filters('wcsf_question_data', $output, $question);
    }

    /**
     * Return a question by id
     *
     * @param integer $id
     *
     * @return object
     */
    public static function get($id)
    {
        if (!is_numeric($id)) {
            return apply_filters('wcsf_question', (object) [], $id);
        }

        return apply_filters('wcsf_question', self::getData(get_post($id)), $id);
    }

    /**
     * Return array of questions
     *
     * @param integer $filterId
     * @param array $ids
     *
     * @return array
     */
    public static function getList($filterId, $ids)
    {
        $ids = array_filter($ids);

        if (empty($filterId) || empty($ids)) {
            return apply_filters('wcsf_questions_list', [], $filterId, $ids);
        }

        global $wpdb;

        // using of an SQL query avoids cross wp_queries problems
        $output = [];
        $questionsIdsString = esc_sql(implode(', ', $ids));
        $queryString = "SELECT $wpdb->posts.* 
            FROM $wpdb->posts
            WHERE $wpdb->posts.post_status = 'publish' 
            AND $wpdb->posts.post_type = 'wcsf_question'
            AND $wpdb->posts.ID IN ($questionsIdsString)
            ORDER BY FIELD(ID, $questionsIdsString) ASC";

        $questions = $wpdb->get_results($queryString, OBJECT);

        foreach ($questions as $question) {
            if (!$question || !property_exists($question, 'ID')) {
                continue;
            }

            $output[$question->ID] = self::getData($question);
        }

        return apply_filters('wcsf_questions_list', $output, $filterId, $ids);
    }

    /**
     * Returns array of ids and titles of posts
     *
     * @param array $args
     *
     * @return array
     */
    public static function getPostsIdsList($args = [])
    {
        $defaults = [
            'post_type' => 'wcsf_question',
            'numberposts' => -1,
            'post_status' => 'publish'
        ];

        $args = array_replace($defaults, $args);
        $output = [];
        $posts = get_posts($args);

        foreach ($posts as $post) {
            $output[$post->ID] = $post->post_title;
        }

        return $output;
    }

    /**
     * Returns array of ids and titles of categories
     *
     * @param array $args
     *
     * @return array
     */
    public static function getCategoriesIdsList($args = [])
    {
        $defaults = [
            'taxonomy' => 'wcsf_question_cat',
            'fields' => 'id=>name',
            'hide_empty' => false,
        ];

        $args = array_replace($defaults, $args);
        $output = get_terms($args);

        return apply_filters('wcsf_questions_categories_list', $output);
    }

    // </editor-fold>

    // <editor-fold desc="Fields">
    /**
     * Get fields according the question settings
     *
     * @param integer $filterId
     * @param integer $questionId
     *
     * @return array
     */
    public static function getFields($filterId, $questionId)
    {
        if (!is_numeric($questionId)) {
            return [];
        }

        $output = [];
        $minMax = [];
        $products = [];
        $component = 'wcsf-field-input';
        $step = self::get($questionId);
        $value = Filter::getValue($filterId);
        $currentValue = isset($value[$questionId]) ? $value[$questionId] : [];

        if (!isset($step->viewType)) {
            return apply_filters('wcsf_question_fields', $output, $filterId, $questionId);
        }

        if (strpos($step->viewType, 'number') !== false) {
            $unit = self::getSetting($questionId, 'numeric_unit');

            // get min/max values
            switch ($step->filterType) {
                case 'price':
                    $minMax = Utils::getMinMaxPrices($products);
                    break;

                case 'attribute':
                    $minMax = Utils::getMinMaxAttributeValues($step->attribute, $products);
                    break;

                case 'meta':
                    $minMax = Utils::getMinMaxMetaValues($step->metaKey, $products);
                    break;
            }

            if ($step->viewType == 'number') {
                $output[] = (object) [
                    'label' => $unit,
                    'name' => "value[$questionId][]",
                    'unit' => $unit,
                    'value' => !empty($currentValue) ? reset($currentValue) : $minMax['min'],
                    'min' => isset($minMax['min']) ? $minMax['min'] : '',
                    'max' => isset($minMax['max']) ? $minMax['max'] : '',
                    'step' => 1,
                    'component' => $component
                ];
            }

            if ($step->viewType == 'number-between') {
                $output['from'] = (object) [
                    'label' => self::getSetting($questionId, 'from_value_text'),
                    'name' => "value[$questionId][from]",
                    'unit' => $unit,
                    'value' => isset($currentValue['from']) ? $currentValue['from'] : $minMax['min'],
                    'min' => isset($minMax['min']) ? $minMax['min'] : '',
                    'max' => isset($minMax['max']) ? $minMax['max'] : '',
                    'step' => 1,
                    'component' => $component
                ];

                $output['to'] = (object) [
                    'label' => self::getSetting($questionId, 'to_value_text'),
                    'name' => "value[$questionId][to]",
                    'unit' => $unit,
                    'value' => isset($currentValue['to']) ? $currentValue['to'] : $minMax['max'],
                    'min' => isset($minMax['min']) ? $minMax['min'] : '',
                    'max' => isset($minMax['max']) ? $minMax['max'] : '',
                    'step' => 1,
                    'component' => $component
                ];
            }
        } else {
            if (in_array($step->filterType, ['tag', 'category', 'attribute'])) {
                switch ($step->filterType) {
                    case 'tag':
                        $taxonomy = 'product_tag';
                        break;

                    case 'attribute':
                        $taxonomy = \wc_attribute_taxonomy_name_by_id((int) $step->attribute);
                        break;

                    default:
                    case 'category':
                        $taxonomy = 'product_cat';
                        break;
                }

                $queryArgs = [
                    'taxonomy' => $taxonomy,
                    'hide_empty' => self::getSetting($questionId, 'hide_empty_terms')
                ];

                if ($step->filterType == 'category') {
                    $queryArgs['parent'] = (int) self::getSetting($questionId, 'category_parent');
                }

                $queryArgs = apply_filters('wcsf_question_fields_query_args', $queryArgs, $filterId, $questionId);
                $terms = Utils::getTermsTree(
                    $queryArgs,
                    [
                        'flatten' => !in_array($step->viewType, ['checkbox']),
                        'showLevelInName' => $step->viewType == 'select',
                    ]
                );

                $output = array_merge($output, self::getTermsFields($terms, $taxonomy, $currentValue, $component));
            } elseif ($step->filterType == 'manual') {
                // prepare manual values
                foreach ($step->manualValues as $valueKey => $value) {
                    $isChecked = $currentValue
                        ? in_array($valueKey, $currentValue)
                        : (isset($value['pre-checked']) ? (bool) $value['pre-checked'] : false);

                    $output[] = (object) [
                        'checked' => $isChecked,
                        'disabled' => false,
                        'name' => $value['name'],
                        'value' => $valueKey,
                        'component' => $component
                    ];
                }
            } elseif ($step->filterType == 'meta') {
                // get meta values
                $values = Utils::getMetaValues($step->metaKey);

                foreach ($values as $productId => $value) {
                    $isChecked = $currentValue ? in_array($value, $currentValue) : false;
                    $output[] = (object) [
                        'checked' => $isChecked,
                        'disabled' => false,
                        'name' => $value,
                        'value' => $value,
                        'component' => $component
                    ];
                }
            }
        }

        return apply_filters('wcsf_question_fields', $output, $filterId, $questionId);
    }

    /**
     * Get fields by terms array
     *
     * @param array $terms
     * @param string $taxonomy
     * @param array $current
     * @param string $component
     *
     * @return array
     */
    public static function getTermsFields($terms, $taxonomy, $current, $component) {
        $output = [];

        foreach ($terms as $term) {
            if (!is_object($term)) {
                continue;
            }

            $isPreChecked = (bool) get_term_meta($term->term_id, '_wcsf_pre_checked', true);
            $isChecked = $current ? in_array($term->term_id, $current) : $isPreChecked;
            $outputItem = (object) [
                'checked' => $isChecked,
                'currentValue' => $current,
                'disabled' => false,
                'name' => $term->name,
                'count' => $term->count,
                'value' => $term->term_id,
                'component' => $component
            ];

            if (isset($term->children) && !empty($term->children)) {
                $outputItem->children = self::getTermsFields($term->children, $taxonomy, $current, $component);
            }

            $output[] = $outputItem;
        }

        return $output;
    }
    // </editor-fold>
}
