<?php
namespace WCStepFilter;

/**
 * Utils Class
 *
 * @class Utils
 * @version 1.0.0
 */
class Utils
{
    /**
     * Build URL from parts
     *
     * @param array $parts
     *
     * @return string
     */
    public static function buildUrl($parts)
    {
        $scheme = isset($parts['scheme']) ? ($parts['scheme'] . '://') : '';
        $host = isset($parts['host']) ? $parts['host'] : '';
        $port = isset($parts['port']) ? (':' . $parts['port']) : '';
        $user = isset($parts['user']) ? $parts['user'] : '';
        $pass = isset($parts['pass']) ? (':' . $parts['pass']) : '';
        $pass = ($user || $pass) ? ($pass . '@') : '';
        $path = isset($parts['path']) ? $parts['path'] : '';
        $query = isset($parts['query']) ? ('?' . $parts['query']) : '';
        $fragment = isset($parts['fragment']) ? ('#' . $parts['fragment']) : '';

        return implode('', [$scheme, $user, $pass, $host, $port, $path, $query, $fragment]);
    }

    /**
     * Get all unique products prices array
     *
     * @param array $ids - specific products ids or nothing
     *
     * @return array
     */
    public static function getProductsPrices($ids = [])
    {
        global $wpdb;

        $sql = "SELECT DISTINCT pm.meta_value FROM "
            . "{$wpdb->postmeta} as pm WHERE pm.meta_key = '_price' ORDER BY pm.meta_value + 0";

        if (!empty($ids)) {
            $inQuery = ' AND pm.post_id IN (' . implode(', ', array_fill(0, count($ids), '%s')) . ')';
            $sql .= call_user_func_array([$wpdb, 'prepare'], array_merge([$inQuery], $ids));
        }

        return $wpdb->get_col($sql);
    }

    /**
     * Get min and max prices of all products
     *
     * @param array $ids - specific products ids or nothing
     *
     * @return array
     */
    public static function getMinMaxPrices($ids = [])
    {
        global $wpdb;

        $sql = 'SELECT MIN(CAST(pm.meta_value as UNSIGNED)) as min, MAX(CAST(pm.meta_value as UNSIGNED)) as max FROM '
            . "{$wpdb->postmeta} as pm WHERE pm.meta_key = '_price'";

        if (!empty($ids)) {
            $inQuery = ' AND pm.post_id IN (' . implode(', ', array_fill(0, count($ids), '%s')) . ')';
            $sql .= call_user_func_array([$wpdb, 'prepare'], array_merge([$inQuery], $ids));
        }

        $results = $wpdb->get_results($sql);

        if (empty($results)) {
            return [
                'min' => null,
                'max' => null
            ];
        }

        $result = array_shift($results);

        return apply_filters(
            'wcsf_min_max_prices',
            [
                'min' => (float) $result->min,
                'max' => (float) $result->max
            ]
        );
    }

    /**
     * Get min and max numeric attribute values
     *
     * @param integer $id - attribute id
     * @param array $ids - specific products ids or nothing
     *
     * @return array
     */
    public static function getMinMaxAttributeValues($id, $ids = [])
    {
        $attributeTaxonomy = wc_attribute_taxonomy_name_by_id((int) $id);
        $terms = get_terms($attributeTaxonomy);

        if (!empty($ids)) {
            // filter by products ids
            foreach ($terms as $key => $term) {
                $enabled = false;

                foreach ($ids as $productId) {
                    if (has_term($term->term_id, $attributeTaxonomy, $productId)) {
                        $enabled = true;

                        break;
                    }
                }

                if (!$enabled) {
                    unset($terms[$key]);
                }
            }
        }

        $values = array_map(function ($o) {
            return (float) $o->name;
        }, $terms);

        return apply_filters(
            'wcsf_min_max_attribute_values',
            [
                'min' => !empty($values) ? (float) min($values) : 0,
                'max' => !empty($values) ? (float) max($values) : 0
            ],
            $id
        );
    }

    /**
     * Get min and max numeric meta values
     *
     * @param string $key
     * @param array $ids - specific products ids or nothing
     *
     * @return array
     */
    public static function getMinMaxMetaValues($key, $ids = [])
    {
        if (empty($key)) {
            return [
                'min' => null,
                'max' => null
            ];
        }

        global $wpdb;

        $sql = $wpdb->prepare(
            'SELECT MIN(CAST(pm.meta_value as UNSIGNED)) as min, MAX(CAST(pm.meta_value as UNSIGNED)) as max FROM '
            . "{$wpdb->postmeta} as pm WHERE pm.meta_key = '%s'",
            $key
        );

        if (!empty($ids)) {
            $inQuery = ' AND pm.post_id IN (' . implode(', ', array_fill(0, count($ids), '%s')) . ')';
            $sql .= call_user_func_array([$wpdb, 'prepare'], array_merge([$inQuery], $ids));
        }

        $results = $wpdb->get_results($sql);

        if (empty($results)) {
            return [
                'min' => null,
                'max' => null
            ];
        }

        $result = array_shift($results);

        return apply_filters(
            'wcsf_min_max_meta_values',
            [
                'min' => (float) $result->min,
                'max' => (float) $result->max
            ]
        );
    }

    /**
     * Get possible meta values array
     *
     * @param string $key
     *
     * @return array
     */
    public static function getMetaValues($key)
    {
        if (empty($key)) {
            return [];
        }

        global $wpdb;

        $output = [];
        $sql = $wpdb->prepare(
            'SELECT DISTINCT pm.post_id, pm.meta_value FROM '
            . "{$wpdb->postmeta} as pm WHERE pm.meta_key = '%s' ORDER BY pm.meta_value",
            $key
        );

        $results = $wpdb->get_results($sql);

        foreach ($results as $result) {
            $output[$result->post_id] = $result->meta_value;
        }

        $output = array_unique($output);

        asort($output);

        return apply_filters('wcsf_meta_values', $output, $key);
    }

    /**
     * Get multi-dimension array of the taxonomy terms
     *
     * @param array $queryArgs
     * @param array $args
     *
     * @return array
     */
    public static function getTermsTree($queryArgs = [], $args = [])
    {
        $output = [];
        $defaults = [
            'flatten' => false,
            'showLevelInName' => false,
            'nameOnly' => false,
            'level' => 0,
            'maxLevel' => 20,
            'startFromLevel' => 0
        ];

        $args = array_replace($defaults, $args);

        if ($args['level'] > $args['maxLevel']) {
            return $output;
        }

        $defaultQuery = [
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
            'hierarchical' => false,
            'parent' => 0
        ];

        $queryArgs = array_replace($defaultQuery, $queryArgs);
        $terms = get_terms($queryArgs);

        foreach ($terms as $term) {
            if (!is_object($term)) {
                continue;
            }

            $term->level = $args['level'];
            $children = self::getTermsTree(
                array_replace($queryArgs, ['parent' => $term->term_id]),
                array_replace($args, ['level' => $args['level'] + 1])
            );

            if ($args['showLevelInName'] && $args['level'] - $args['startFromLevel'] >= 0) {
                $term->name = str_repeat('—', $args['level'] - $args['startFromLevel']) . $term->name;
            }

            if ($args['flatten']) {
                if ($args['level'] >= $args['startFromLevel']) {
                    $output[$term->term_id] = $args['nameOnly'] ? $term->name : $term;
                }

                $output = array_replace($output, $children);

                continue;
            }

            if ($args['level'] >= $args['startFromLevel']) {
                if (!empty($children)) {
                    $term->children = $children;
                }

                $output[$term->term_id] = $args['nameOnly'] ? $term->name : $term;
            } else {
                $output = array_replace($output, $children);
            }
        }

        return $output;
    }

    /**
     * Parse JSONed request to an array
     *
     * @param array $postData
     *
     * @return array
     */
    public static function parseArrayOfJSONs($postData)
    {
        foreach ($postData as &$value) {
            if (!is_string($value)) {
                continue;
            }

            $decode = json_decode(stripslashes($value), true);
            $value = $decode ? $decode : $value;
        }

        return $postData;
    }

    /**
     * Send a JSON request
     *
     * @param array $data
     */
    public static function sendJSON($data)
    {
        wp_send_json(apply_filters('wcsf_send_json_data', $data));
    }
}
