<?php
namespace WCStepFilter;

/**
 * Core Class
 *
 * @class Core
 * @version 4.14.0
 */
class Core
{
    // <editor-fold desc="Properties">
    /**
     * Self instance variable
     * @var Core The single instance of the class
     */
    protected static $instance = null;

    /**
     * Is WooCommerce active
     * @var bool
     */
    public static $wcIsActive = false;

    /**
     * Plugin global settings model
     *
     * @var array
     */
    public static $settingsModel;

    /**
     * Array of the showed templates
     * @var array
     */
    public static $templatesShowed = [];

    /**
     * Array of the showing template args
     * @var array
     */
    public static $currentTemplateArgs = [];

    /**
     * Filter class instance
     *
     * @var object
     */
    public $filter;

    /**
     * Question class instance
     *
     * @var object
     */
    public $question;
    // </editor-fold>

    // <editor-fold desc="Core">
    /** Constructor */
    public function __construct()
    {
        // include base slave classes
        $requiredClasses = [
            'L10N',
            'Utils',
            'Storage',
            'Filter',
            'Question'
        ];

        foreach ($requiredClasses as $requiredClass) {
            if (!class_exists('\\WCStepFilter\\' . $requiredClass)
                && file_exists(__DIR__ . DIRECTORY_SEPARATOR . $requiredClass . '.php')
            ) {
                include_once(__DIR__ . DIRECTORY_SEPARATOR . $requiredClass . '.php');
            }
        }

        // actions
        add_action('init', [$this, 'onInit']);
        add_action('plugins_loaded', [$this, 'pluginsLoadedHook']);
        add_action('woocommerce_init', [$this, 'onWoocommerceInit'], 1);
        add_action('wp_loaded', [$this, 'requests']);
        add_action('plugins_loaded', [$this, 'loadTextDomain']);
        add_filter('plugin_action_links', [$this, 'actionLinksFilter'], 10, 2);
        add_filter('plugin_row_meta', [$this, 'metaLinksFilter'], 10, 2);

        if (isset($_GET['wcsf-results'])) {
            add_action('woocommerce_product_query', [$this, 'modifyProductsQuery'], 15);
        }

        // settings page
        add_filter('woocommerce_settings_tabs_array', [$this, 'addSettingsTab'], 50);
        add_action('woocommerce_settings_tabs_step_filter_settings_tab', [$this, 'initSettingsTab']);
        add_action('woocommerce_update_options_step_filter_settings_tab', [$this, 'updateSettingsTab']);

        // scripts and styles
        add_action('wp_enqueue_scripts', [$this, 'enqueueFront']);
        add_action('admin_enqueue_scripts', [$this, 'enqueueAdmin']);

        // init basic classes
        $this->filter = new Filter();
        $this->question = new Question();
    }

    /**
     * Get single class instance
     *
     * @static
     * @return Core
     */
    public static function instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /** On init action */
    public function onInit()
    {
        self::$settingsModel = apply_filters(
            'wcsf_global_settings_model',
            [
                'store_session_in_db' => [
                    'name' => L10N::r('Store session in the DB'),
                    'default' => 'no',
                    'type' => 'checkbox',
                    'desc' => L10N::r('Tick in a case of problems of the filter\'s state storing'),
                    'id' => 'woocommerce_step_filter_store_session_in_db',
                    'key' => 'woocommerce_step_filter_store_session_in_db'
                ]
            ]
        );
    }

    /** Handles then plugins are loaded */
    public function pluginsLoadedHook()
    {
        self::$wcIsActive = class_exists('\WooCommerce');

        if (!self::$wcIsActive) {
            add_action('admin_notices', function () {
                echo '<div class="notice notice-warning is-dismissible"><p>'
                    . L10N::r('WooCommerce is required for WC Step Filter')
                    . '</p></div>';
            });
        }
    }

    /** Runs after woocommerce is loaded */
    public function onWoocommerceInit()
    {
        add_action('wcsf_shortcode_init', [$this, 'shortCodeAction']);
    }

    /**
     * Short-code call action
     *
     * @param array $attributes
     */
    public function shortCodeAction($attributes)
    {
        // start WC session variable if needed
        if (self::getSetting('store_session_in_db')) {
            WC()->initialize_session();
            WC()->session->set_customer_session_cookie(true);
        }

        if (!isset($attributes['id']) || empty($attributes['id'])) {
            return;
        }

        $id = (int) $attributes['id'];
        $resultsUrl = $this->filter->getSetting($id, 'results_url');

        if ($resultsUrl == '#') {
            if (!isset($_GET['wcsf-results']) || empty($_GET['wcsf-results'])) {
                // reset filter's state if it's results page is any and there is no query parameters
                $this->filter->resetState($id);
            } elseif (isset($_GET['wcsf-value'], $_GET['wcsf-results']) && !empty($_GET['wcsf-value'])
                && $_GET['wcsf-results'] == $id
            ) {
                try {
                    // set filter's value from get parameter
                    $this->filter->setValue($id, $_GET['wcsf-value']);
                } catch (\Exception $exception) {
                    $this->filter->addNotice(
                        $exception->getCode() ? $exception->getCode() : $this->filter->getActiveStepId($id),
                        ['message' => $exception->getMessage()]
                    );
                }
            }
        }
    }

    /** Load text domain */
    public function loadTextDomain()
    {
        load_plugin_textdomain(
            'woocommerce-step-filter',
            false,
            basename(WC_STEP_FILTER_PLUGIN_PATH) . '/languages/'
        );
    }

    /**
     * Plugins list meta link filter
     *
     * @param array $links
     * @param string $plugin
     *
     * @return array
     */
    public function metaLinksFilter($links, $plugin)
    {
        if (false === strpos($plugin, 'woocommerce-step-filter.php')) {
            return $links;
        }

        $extraLinks = [
            'docs' => '<a href="https://step-filter.troll-winner.ru/docs/" target="_blank" rel="nofollow noopener">'
                . L10N::r('Docs') . '</a>'
        ];

        $links = array_merge($links, $extraLinks);

        return $links;
    }

    /**
     * Plugins list action links filter
     *
     * @param array $links
     * @param string $plugin
     *
     * @return array
     */
    public function actionLinksFilter($links, $plugin)
    {
        if (false === strpos($plugin, 'woocommerce-step-filter.php')) {
            return $links;
        }

        $extraLinks = [
            'upgrade' => '<a href="http://step-filter.troll-winner.ru/plugin-shop-page/" '
                . 'target="_blank" rel="nofollow noopener">' . L10N::r('More features') . '</a>',
            'settings' => '<a href="admin.php?page=wc-settings&tab=step_filter">'
                . L10N::r('Settings') . '</a>'
        ];

        $links = array_merge($extraLinks, $links);

        return $links;
    }
    // </editor-fold>

    // <editor-fold desc="HTML">
    /**
     * Include php-template by the name.
     * First looking in the "theme folder/woocommerce-step-filter (WC_STEP_FILTER_THEME_TEMPLATES_DIR)"
     * Second looking in the "plugin folder/views"
     * Making extraction of the arguments as variables
     *
     * @param string $name
     * @param array  $arguments
     * @param array  $settings
     *
     * @return string
     */
    public static function getTemplatePart($name = '', $arguments = [], $settings = [])
    {
        $defaultSettings = [
            'echo' => true,
            'once' => false
        ];

        $settings = array_merge($defaultSettings, $settings);

        // show template only once
        if ($settings['once'] && in_array($name, self::$templatesShowed)) {
            return null;
        }

        // save template as showed
        self::$templatesShowed[] = $name;

        if (is_array($arguments)) {
            extract($arguments, EXTR_PREFIX_SAME, 'data');

            self::$currentTemplateArgs = $arguments;
        }

        $path = get_stylesheet_directory() . DIRECTORY_SEPARATOR
            . WC_STEP_FILTER_THEME_TEMPLATES_DIR . DIRECTORY_SEPARATOR . $name . '.php';

        if (!file_exists($path)) {
            $path = WC_STEP_FILTER_PLUGIN_PATH . DIRECTORY_SEPARATOR
                . 'views' . DIRECTORY_SEPARATOR . $name . '.php';
        }

        if (!file_exists($path)) {
            return '';
        }

        ob_start();

        include($path);

        $html = ob_get_clean();

        if ($settings['echo']) {
            echo $html;

            return null;
        }

        return $html;
    }

    /**
     * Get requested HTML part arguments
     *
     * @param array $defaults
     * @param array $settings
     *
     * @return array
     */
    public static function getTemplatePartArgs($defaults = [], $settings = [])
    {
        $defaultsSettings = ['recursive' => false];
        $settings = array_replace($defaultsSettings, $settings);
        $arguments = self::$currentTemplateArgs;

        if (!empty($arguments)) {
            foreach ($defaults as $defaultKey => $_) {
                // find arguments from shortcode attributes
                if (strtolower($defaultKey) == $defaultKey || !isset($arguments[strtolower($defaultKey)])) {
                    continue;
                }

                $arguments[$defaultKey] = $arguments[strtolower($defaultKey)];

                unset($arguments[strtolower($defaultKey)]);
            }

            if ($settings['recursive']) {
                $arguments = array_replace_recursive($defaults, $arguments);
            } else {
                $arguments = array_replace($defaults, $arguments);
            }

            return $arguments;
        }

        return $defaults;
    }
    // </editor-fold>

    // <editor-fold desc="Query">
    /** Add request actions */
    public function requests()
    {
        if (is_admin()) {
            return;
        }

        // get queries
        // reset the filter
        if (!isset($_POST['results']) && isset($_GET['wcsf-reset-state'])) {
            $this->filter->resetState((int) $_GET['wcsf-reset-state']);
        }

        // reset one value
        if (!isset($_POST['results']) && isset($_GET['wcsf-reset-value'], $_GET['wcsf-results'])) {
            $this->filter->resetValue((int) $_GET['wcsf-results'], (int) $_GET['wcsf-reset-value']);
        }

        // post queries
        if (!isset($_POST['wcsf'])) {
            return;
        }

        $id = isset($_POST['id']) ? (int) $_POST['id'] : false;

        // skip step
        if (isset($_POST['skip'])) {
            $this->filter->setActiveStep($id, esc_sql($_POST['skip']));

            return;
        }

        // back step
        if (isset($_POST['back'])) {
            $this->filter->setActiveStep($id, esc_sql($_POST['back']));

            return;
        }

        // save value
        if (isset($_POST['value']) && !empty($_POST['value'])) {
            try {
                $this->filter->setValue($id, (array) $_POST['value']);
            } catch (\Exception $exception) {
                $this->filter->addNotice(
                    $exception->getCode() ? $exception->getCode() : $this->filter->getActiveStepId($id),
                    ['message' => $exception->getMessage()]
                );
            }
        }

        // change step
        if (isset($_POST['stepId'])) {
            $this->filter->setActiveStep($id, esc_sql($_POST['stepId']));
        }

        // reset filter
        if (isset($_POST['reset'])) {
            $this->filter->resetState($id);
        }

        // show results page
        if (isset($_POST['results'])) {
            $redirectLink = $this->filter->getResultsUrl($id);
            $redirectLink = apply_filters('wcsf_results_redirect_link', $redirectLink, $id);
            $redirectCode = apply_filters('wcsf_results_redirect_code', 302, $id, $redirectLink);

            do_action('wcsf_before_results_redirect', $id, $redirectLink);

            wp_redirect($redirectLink, $redirectCode);

            // js version of redirect
            exit("<script>document.location = '$redirectLink';</script>");
        }
    }

    /**
     * Get query args by the filter id and value
     * @param integer $id
     * @param array $value
     * @param array $args
     *
     * @return array
     */
    public static function getQueryArgsByFilter($id, $value, $args = [])
    {
        $taxQuery = [];
        $postIn = [];
        $postNotIn = [];
        $metaQuery = [];
        $multiChoiceOperators = apply_filters(
            'wsf_multi_choice_operators',
            [
                -1 => 'NOT IN',
                1 => 'IN',
                2 => 'AND'
            ]
        );

        // blend in passed category or tag id
        if (isset($args['categoryId']) && $args['categoryId']) {
            $taxQuery[] = [
                'taxonomy' => 'product_cat',
                'field' => 'id',
                'terms' => $args['categoryId']
            ];
        } elseif (isset($args['tagId']) && $args['tagId']) {
            $taxQuery[] = [
                'taxonomy' => 'product_tag',
                'field' => 'id',
                'terms' => $args['tagId']
            ];
        }

        // loop for the each filter value
        foreach ((array) $value as $questionId => $questionValue) {
            $question = Question::get($questionId);

            if (!$question) {
                continue;
            }

            $viewType = $question->viewType;
            $filterType = $question->filterType;
            $relation = isset($question->relation) ? $question->relation : 'or';
            $operator = $relation == 'or' ? 'IN' : 'AND';

            if (in_array($filterType, ['tag', 'category', 'attribute'])) {
                // attribute, category or tag
                if ($filterType == 'attribute') {
                    // filtering by attribute
                    if (!isset($question->attribute)) {
                        continue;
                    }

                    $taxonomy = \wc_attribute_taxonomy_name_by_id((int) $question->attribute);
                } else {
                    // filtering by tag or category
                    $taxonomy = $filterType == 'tag' ? 'product_tag' : 'product_cat';
                }

                if (strpos($viewType, 'number') !== false) {
                    // numeric type of field
                    $filterBy = [];
                    $terms = get_terms(['taxonomy' => $taxonomy]);

                    foreach ($terms as $term) {
                        $termValue = (float) $term->name;
                        $issetFrom = isset($questionValue['from']) && !empty($questionValue['from']);
                        $from = $issetFrom && $questionValue['from'] <= $termValue;
                        $issetTo = isset($questionValue['to']) && !empty($questionValue['to']);
                        $to = $issetTo && $questionValue['to'] >= $termValue;

                        // check by numeric with the first, the second and both values
                        if (($viewType == 'number' && !empty($questionValue))
                            || ($viewType == 'number-between'
                                && ($from && $to || !$issetTo && $from || !$issetFrom && $to))
                        ) {
                            $filterBy[] = $term->term_id;
                        }
                    }

                    if (!empty($filterBy)) {
                        $taxQuery[] = [
                            'taxonomy' => $taxonomy,
                            'field' => 'id',
                            'terms' => $filterBy,
                            'operator' => 'IN'
                        ];
                    }
                } else {
                    if (strpos($viewType, 'multi-choice') !== false) {
                        $in = [];

                        foreach (array_filter($questionValue) as $termId => $choiceValue) {
                            if ($choiceValue == 1) {
                                // get all IN values in the array
                                $in[] = $termId;

                                continue;
                            }

                            $taxQuery[] = [
                                'taxonomy' => $taxonomy,
                                'field' => 'id',
                                'terms' => $termId,
                                'operator' => $multiChoiceOperators[$choiceValue]
                            ];
                        }

                        if (!empty($in)) {
                            $taxQuery[] = [
                                'taxonomy' => $taxonomy,
                                'field' => 'id',
                                'terms' => $in,
                                'operator' => $multiChoiceOperators[1]
                            ];
                        }
                    } elseif (!empty(array_filter($questionValue))) {
                        // skip "doesn't matter" and not-defined value and filter by other terms
                        $taxQuery[] = [
                            'taxonomy' => $taxonomy,
                            'field' => 'id',
                            'terms' => array_filter($questionValue),
                            'operator' => $operator
                        ];
                    }
                }
            } elseif ($filterType == 'manual') {
                // filtering by a manual value
                foreach ($questionValue as $questionValueItem) {
                    if (!isset($question->manualValues[$questionValueItem])) {
                        continue;
                    }

                    $filterBy = $question->manualValues[$questionValueItem]['filter-by'];
                    $items = isset($question->manualValues[$questionValueItem]['items'])
                        ? $question->manualValues[$questionValueItem]['items']
                        : [];

                    $items = !empty($items) ? $items : [-1]; // if nothing is selected

                    if ($filterBy == 'any') {
                        continue;
                    } elseif ($filterBy == 'tag') {
                        $taxQuery[] = [
                            'taxonomy' => 'product_tag',
                            'field' => 'id',
                            'terms' => array_filter($items, function ($item) {
                                return is_numeric($item);
                            }),
                            'operator' => $operator
                        ];
                    } elseif ($filterBy == 'product') {
                        if (empty($postIn)) {
                            $postIn = $items;
                        } else {
                            $postIn = array_intersect($postIn, $items);
                        }
                    }
                }
            } elseif (in_array($filterType, ['price', 'meta'])) {
                // filtering by price
                $metaKey = $filterType == 'price' ? '_price' : $question->metaKey;
                $metaValue = null;
                $compare = '=';
                $type = 'CHAR';

                if (strpos($viewType, 'number') !== false) {
                    $type = 'NUMERIC';

                    switch ($viewType) {
                        case 'number':
                            $compare = '=';
                            $metaValue = !empty($questionValue) ? reset($questionValue) : null;

                            break;
                        case 'number-between':
                            if (isset($questionValue['from']) && !empty($questionValue['from'])
                                && isset($questionValue['to'])
                                && !empty($questionValue['to'])
                            ) {
                                $compare = 'BETWEEN';
                                $metaValue = array_values($questionValue);
                            } elseif (isset($questionValue['from']) && !empty($questionValue['from'])) {
                                $compare = '>=';
                                $metaValue = $questionValue['from'];
                            } elseif (isset($questionValue['to']) && !empty($questionValue['to'])) {
                                $compare = '<=';
                                $metaValue = $questionValue['to'];
                            }
                    }
                } elseif (strpos($viewType, 'multi-choice') !== false) {
                    $in = [];

                    foreach (array_filter($questionValue) as $metaValue => $choiceValue) {
                        if ($choiceValue == 1) {
                            // get all IN values in the array
                            $in[] = $metaValue;

                            continue;
                        }

                        $metaQuery[] = [
                            'key' => $metaKey,
                            'value' => $metaValue,
                            'compare' => $multiChoiceOperators[$choiceValue],
                            'type' => $type
                        ];
                    }

                    if (!empty($in)) {
                        $metaQuery[] = [
                            'key' => $metaKey,
                            'value' => $in,
                            'compare' => $multiChoiceOperators[1],
                            'type' => $type
                        ];
                    }

                    $metaValue = null;
                } else {
                    $metaValue = $questionValue;
                }

                if (!is_null($metaValue)) {
                    $metaQuery[] = [
                        'key' => $metaKey,
                        'value' => $metaValue,
                        'compare' => $compare,
                        'type' => $type
                    ];
                }
            }
        }

        return apply_filters(
            'wcsf_query_args_by_filter',
            [
                'tax_query' => $taxQuery,
                'meta_query' => $metaQuery,
                'post__in' => $postIn,
                'post__not_in' => $postNotIn
            ],
            $id,
            $value,
            $args
        );
    }

    /**
     * Filter the products on the shop page
     *
     * @param array $query
     *
     * @return null
     */
    public function modifyProductsQuery($query)
    {
        if (is_admin() || !isset($_GET['wcsf-results']) || !isset($_GET['wcsf-value'])) {
            return;
        }

        do_action('wcsf_before_modify_request', $query);

        $id = (int) $_GET['wcsf-results'];
        $value = (array) $_GET['wcsf-value'];
        $args = $this->getQueryArgsByFilter($id, $value);

        if (method_exists($query, 'set')) {
            if (!empty($args['tax_query'])) {
                $query->set('tax_query', $args['tax_query']);
            }

            if (!empty($args['post__in'])) {
                $query->set('post__in', $args['post__in']);
            }

            if (!empty($args['post__not_in'])) {
                $query->set('post__not_in', $args['post__not_in']);
            }

            if (!empty($args['meta_query'])) {
                $query->set('meta_query', $args['meta_query']);
            }
        }

        do_action('wcsf_after_modify_request', $query);

        // remove filter
        remove_action('woocommerce_product_query', [$this, 'modifyProductsQuery']);

        return;
    }
    // </editor-fold>

    // <editor-fold desc="Assets">
    /** Admin styles and scripts enqueue */
    public function enqueueAdmin()
    {
        wp_enqueue_media();
        wp_enqueue_script('jquery');
        wp_enqueue_script('jquery-ui-sortable');
        wp_enqueue_script(
            'woocommerce-step-filter-data-table',
            WC_STEP_FILTER_PLUGIN_URL . 'assets/admin/js/data-table.min.js',
            [
                'jquery',
                'jquery-ui-sortable'
            ],
            WC_STEP_FILTER_VERSION,
            true
        );

        wp_enqueue_script(
            'woocommerce-step-filter-multi-select',
            WC_STEP_FILTER_PLUGIN_URL . 'assets/admin/js/multi-select.min.js',
            ['jquery'],
            WC_STEP_FILTER_VERSION,
            true
        );

        wp_enqueue_script(
            'woocommerce-step-filter-hooks',
            WC_STEP_FILTER_PLUGIN_URL . 'assets/admin/js/hooks.min.js',
            ['jquery'],
            WC_STEP_FILTER_VERSION,
            true
        );

        wp_enqueue_style(
            'woocommerce-step-filter',
            WC_STEP_FILTER_PLUGIN_URL . 'assets/admin/css/app.min.css',
            [],
            WC_STEP_FILTER_VERSION
        );
    }

    /** Styles and scripts enqueue */
    public function enqueueFront()
    {
        wp_enqueue_script('jquery');
        wp_enqueue_script(
            'woocommerce-step-filter-scripts',
            WC_STEP_FILTER_PLUGIN_URL . 'assets/front/js/scripts.min.js',
            ['jquery'],
            WC_STEP_FILTER_VERSION,
            true
        );

        wp_enqueue_style(
            'woocommerce-step-filter-full',
            WC_STEP_FILTER_PLUGIN_URL . 'assets/front/css/app-full.min.css',
            [],
            WC_STEP_FILTER_VERSION
        );
    }
    // </editor-fold>

    // <editor-fold desc="Settings">
    /**
     * Add the tab to the woocommerce options page
     *
     * @param array $settingsTabs
     *
     * @return array $settingsTabs
     */
    public function addSettingsTab($settingsTabs)
    {
        $settingsTabs['step_filter_settings_tab'] = 'Step Filter';

        return $settingsTabs;
    }

    /** Init options fields */
    public function initSettingsTab()
    {
        $defaults = [
            'section_title' => [
                'name' => 'Step Filter',
                'type' => 'title',
                'desc' => '',
                'id' => 'wc_step_filter_section_title'
            ]
        ];

        $args = array_merge($defaults, self::$settingsModel);

        woocommerce_admin_fields($args);
        woocommerce_admin_fields(
            [
                'section_end' => [
                    'type' => 'sectionend',
                    'id' => 'wc_step_filter_section_end'
                ]
            ]
        );
    }

    /** Save options fields */
    public function updateSettingsTab()
    {
        woocommerce_update_options(self::$settingsModel);
    }

    /**
     * Get one of settings
     *
     * @param string $setting
     *
     * @return string|boolean|array|float
     */
    public static function getSetting($setting)
    {
        if (!isset(self::$settingsModel[$setting])) {
            return apply_filters('wcsf_global_setting', null, $setting);
        }

        static $globalSettingsCache;

        if (isset($globalSettingsCache[$setting])) {
            return apply_filters('wcsf_global_setting', $globalSettingsCache[$setting], $setting);
        }

        $value = get_option(self::$settingsModel[$setting]['key'], null);

        if (is_null($value) && isset(self::$settingsModel[$setting]['default'])) {
            $value = self::handleSettingType(
                self::$settingsModel[$setting]['default'],
                self::$settingsModel[$setting]['type']
            );
        }

        $globalSettingsCache[$setting] = self::handleSettingType($value, self::$settingsModel[$setting]['type']);

        return apply_filters('wcsf_global_setting', $globalSettingsCache[$setting], $setting);
    }

    /**
     * Generate html-field from passed args
     *
     * @param array $modelItem
     * @param array $args
     */
    public static function settingFieldView($modelItem, $args = [])
    {
        $defaultArgs = [
            'values' => [],
            'namePattern' => '%key%',
            'generateId' => true,
            'asTemplate' => false
        ];

        $args = array_replace_recursive($defaultArgs, $args);

        // create name from pattern
        $args['name'] = str_replace('%key%', $modelItem['key'], $args['namePattern']);

        // define value
        $args['value'] = isset($args['values'][$modelItem['key']])
            ? $args['values'][$modelItem['key']]
            : $modelItem['default'];

        // define id attribute
        $args['id'] = $args['generateId'] ? $modelItem['key'] : false;

        $viewPath = __DIR__ . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR
            . 'setting-field-views' . DIRECTORY_SEPARATOR . $modelItem['type'] . '.php';

        echo isset($modelItem['before']) ? $modelItem['before'] : '';

        if (file_exists($viewPath)) {
            include($viewPath);
        }

        echo isset($modelItem['after']) ? $modelItem['after'] : '';

        if (isset($modelItem['description'])) {
            echo '<p class="description">' . wp_kses_post($modelItem['description']) . '</p>';
        }
    }

    /**
     * Handle the setting value according to the type
     *
     * @param mixed $value
     * @param string $type
     *
     * @return string|float|bool|array
     */
    public static function handleSettingType($value, $type = 'string')
    {
        switch ($type) {
            case 'checkbox':
                $value = filter_var($value, FILTER_VALIDATE_BOOLEAN);
                break;

            case 'number':
                $value = (float) $value;
                break;

            case 'array':
                $value = (array) $value;
                break;
        }

        return $value;
    }
    // </editor-fold>
}
